/*
  ******************************************************************************
  * @file    HAL_PMU.h
  * @version V1.0.0
  * @date    2022
  * @brief   Header file of PMU HAL module.
  ******************************************************************************
*/

#ifndef __HAL_PMU_H__
#define __HAL_PMU_H__

#include "acm32g103_hal_conf.h"







/**
  * @brief  IO wake-up source
  */
typedef enum
{
    STANDBY_WAKEUP1  = 0x00000001,    //PA0 
    STANDBY_WAKEUP2  = 0x00000002,    //PC13   
    STANDBY_WAKEUP3  = 0x00000004,    //PA2  
    STANDBY_WAKEUP4  = 0x00000008,    //PC5  
    STANDBY_WAKEUP5  = 0x00000010,    //PB5  
}PMU_WakeUpIo_t;


/**
  * @brief  RTC wake-up source
  */
typedef enum
{
    STANDBY_WAKEUP_WUTIE  = RTC_IE_WUTIE,
    STANDBY_WAKEUP_STAMP2 = (RTC_IE_STP2RIE | RTC_IE_STP2FIE),
    STANDBY_WAKEUP_STAMP1 = (RTC_IE_STP1RIE | RTC_IE_STP1FIE),
    STANDBY_WAKEUP_32S    = RTC_IE_ADJ32,
    STANDBY_WAKEUP_ALARM  = RTC_IE_ALM,
    STANDBY_WAKEUP_1KHZ   = RTC_IE_1KHZ,
    STANDBY_WAKEUP_256HZ  = RTC_IE_256HZ,
    STANDBY_WAKEUP_64HZ   = RTC_IE_64HZ,
    STANDBY_WAKEUP_16HZ   = RTC_IE_16HZ,
    STANDBY_WAKEUP_8HZ    = RTC_IE_8HZ,
    STANDBY_WAKEUP_4HZ    = RTC_IE_4HZ,
    STANDBY_WAKEUP_2HZ    = RTC_IE_2HZ,   
    STANDBY_WAKEUP_SEC    = RTC_IE_SEC,
    STANDBY_WAKEUP_MIN    = RTC_IE_MIN,
    STANDBY_WAKEUP_HOUR   = RTC_IE_HOUR,
    STANDBY_WAKEUP_DATE   = RTC_IE_DATE,
}PMU_WakeUpRtc_t;  


/**
  * @brief  Wake Up polarity Configuration
  */

#define STANDBY_WAKEUP_HIGH                     (0x00000000)
#define STANDBY_WAKEUP_LOW                      (0x00000001)


/**
  * @brief  Wait for an interrupt or event
  */

#define WAIT_FOR_INT                            (0x00000000)  
#define WAIT_FOR_EVENT                          (0x00000001)


/**
  * @brief  Low power mode
  */
#define PMU_CTL0_LPMS_STOP0                     ( 0x0U << PMU_CTL0_LPMS_Pos )  
#define PMU_CTL0_LPMS_STOP1                     ( 0x1U << PMU_CTL0_LPMS_Pos ) 
#define PMU_CTL0_LPMS_STOP2                     ( 0x2U << PMU_CTL0_LPMS_Pos )  
#define PMU_CTL0_LPMS_STANDBY                   ( 0x4U << PMU_CTL0_LPMS_Pos ) 
#define PMU_CTL0_LPMS_POWER_DOWN                ( 0x6U << PMU_CTL0_LPMS_Pos ) 



/** @defgroup PMU LVD detection Voltage 
  * @{
  */ 
#define PMU_LVDVOLTAGE_1V71                     (0U << PMU_CTL1_LVD_SEL_Pos)
#define PMU_LVDVOLTAGE_2V01                     (1U << PMU_CTL1_LVD_SEL_Pos)
#define PMU_LVDVOLTAGE_2V23                     (2U << PMU_CTL1_LVD_SEL_Pos)
#define PMU_LVDVOLTAGE_2V43                     (3U << PMU_CTL1_LVD_SEL_Pos)
#define PMU_LVDVOLTAGE_2V51                     (4U << PMU_CTL1_LVD_SEL_Pos)
#define PMU_LVDVOLTAGE_2V73                     (5U << PMU_CTL1_LVD_SEL_Pos)
#define PMU_LVDVOLTAGE_2V80                     (6U << PMU_CTL1_LVD_SEL_Pos)
#define PMU_LVDVOLTAGE_2V90                     (7U << PMU_CTL1_LVD_SEL_Pos)
#define IS_PMU_LVD_VOLTAGE(VOLTAGE)             (((VOLTAGE) == PMU_LVDVOLTAGE_1V71) || ((VOLTAGE) == PMU_LVDVOLTAGE_2V01)|| \
                                                ((VOLTAGE) == PMU_LVDVOLTAGE_2V23) || ((VOLTAGE) == PMU_LVDVOLTAGE_2V43)|| \
                                                ((VOLTAGE) == PMU_LVDVOLTAGE_2V51) || ((VOLTAGE) == PMU_LVDVOLTAGE_2V73)|| \
                                                ((VOLTAGE) == PMU_LVDVOLTAGE_2V80) || ((VOLTAGE) == PMU_LVDVOLTAGE_2V90))
/**
  * @}
  */
  
/** @defgroup LVD Filter Definitions 
* @{
*/ 
#define PMU_LVDFILTER_1SAMPLE                   (0U << PMU_CTL1_FLT_TIME_Pos)
#define PMU_LVDFILTER_2SAMPLE                   (1U << PMU_CTL1_FLT_TIME_Pos)
#define PMU_LVDFILTER_4SAMPLE                   (2U << PMU_CTL1_FLT_TIME_Pos)                                   
#define PMU_LVDFILTER_16SAMPLE                  (3U << PMU_CTL1_FLT_TIME_Pos)
#define PMU_LVDFILTER_64SAMPLE                  (4U << PMU_CTL1_FLT_TIME_Pos) 
#define PMU_LVDFILTER_256SAMPLE                 (5U << PMU_CTL1_FLT_TIME_Pos) 
#define PMU_LVDFILTER_1024SAMPLE                (6U << PMU_CTL1_FLT_TIME_Pos)
#define PMU_LVDFILTER_4095SAMPLE                (7U << PMU_CTL1_FLT_TIME_Pos)
#define IS_PMU_LVD_FILTER(FILTER)               (((FILTER) == PMU_LVDFILTER_1SAMPLE) || \
                                                ((FILTER) == PMU_LVDFILTER_2SAMPLE) || \
                                                ((FILTER) == PMU_LVDFILTER_4SAMPLE) || \
                                                ((FILTER) == PMU_LVDFILTER_16SAMPLE) || \
                                                ((FILTER) == PMU_LVDFILTER_64SAMPLE) || \
                                                ((FILTER) == PMU_LVDFILTER_256SAMPLE) || \
                                                ((FILTER) == PMU_LVDFILTER_1024SAMPLE) || \
                                                ((FILTER) == PMU_LVDFILTER_4095SAMPLE))
/**
  * @}
  */ 
 
/** @defgroup LVD Filter Enable
* @{
*/ 
#define PMU_LVDFILTER_DISABLE                   (0U << PMU_CTL1_LVD_FLTEN_Pos)
#define PMU_LVDFILTER_ENABLE                    (1U << PMU_CTL1_LVD_FLTEN_Pos) 
#define IS_PMU_LVD_FILTER_STATE(STATE)          (((STATE) == PMU_LVDFILTER_DISABLE) ||\
                                                ((STATE) == PMU_LVDFILTER_ENABLE))
/**
  * @}
  */ 

  
/** @defgroup PMU BOR detection Voltage 
  * @{
  */ 
#define PMU_BORVOLTAGE_2V_2V1                   (0U << PMU_CTL2_BOR_CFG_Pos)
#define PMU_BORVOLTAGE_2V2_2V3                  (1U << PMU_CTL2_BOR_CFG_Pos)
#define PMU_BORVOLTAGE_2V49_2V61                (2U << PMU_CTL2_BOR_CFG_Pos)
#define PMU_BORVOLTAGE_2V77_2V9                 (3U << PMU_CTL2_BOR_CFG_Pos)
#define IS_PMU_BOR_VOLTAGE(VOLTAGE)             (((VOLTAGE) == PMU_BORVOLTAGE_2V_2V1) || \
                                                ((VOLTAGE) == PMU_BORVOLTAGE_2V2_2V3)|| \
                                                ((VOLTAGE) == PMU_BORVOLTAGE_2V49_2V61) || \
                                                ((VOLTAGE) == PMU_BORVOLTAGE_2V77_2V9))
/**
  * @}
  */
  
  

/**
  * @brief  Alarm day or week mode
  */
#define RTC_ALM_ALM_WDS_WEEK                ( 0x0U << RTC_ALM_ALM_WDS_Pos )

/**
  * @brief  Tamper Pin 
  */
#define PMU_IOSEL_PC13_SEL_RTC_TAMPER       ( 0x2U << PMU_IOSEL_PC13_SEL_Pos )


/** @brief  PC13 function select 
  * @param  __FUNC__: PC13 function select.
  *         This parameter can be 0: GPIO1RTC Fout2RTC tamper 3PC13 Value
  */
#define __HAL_PMU_PC13_SEL(__FUNC__)    (PMU->IOSEL |= (PMU->IOSEL & ~(0x3)) | (__FUNC__))

/** @brief  PC14 function select 
  * @param  __FUNC__: PC14 function select.
  *         This parameter can be 0: GPIO1PC14 Value
  */
#define __HAL_PMU_PC14_SEL(__FUNC__)    (PMU->IOSEL |= (PMU->IOSEL & ~(0x3 << 3)) | (__FUNC__ << 3))

/** @brief  PC15 function select 
  * @param  __FUNC__: PC15 function select.
  *         This parameter can be 0: GPIO1PC15 Value
  */
#define __HAL_PMU_PC15_SEL(__FUNC__)    (PMU->IOSEL |= (PMU->IOSEL & ~(0x3 << 5)) | (__FUNC__ << 5))

/** @brief  PC13 Value set 
  * @param  __FUNC__: PC13 Value set.
  *         This parameter can be 0: set1claer
  */
#define __HAL_PMU_PC13_VALUE(__VALUE__)    (PMU->IOSEL |= (PMU->IOSEL & ~(1 << 8)) | (__VALUE__ << 8))

/** @brief  PC14 Value set 
  * @param  __FUNC__: PC14 Value set.
  *         This parameter can be 0: set1claer
  */
#define __HAL_PMU_PC14_VALUE(__VALUE__)    (PMU->IOSEL |= (PMU->IOSEL & ~(1 << 9)) | (__VALUE__ << 9))

/** @brief  PC15 Value set 
  * @param  __FUNC__: PC15 Value set.
  *         This parameter can be 0: set1claer
  */
#define __HAL_PMU_PC15_VALUE(__VALUE__)    (PMU->IOSEL |= (PMU->IOSEL & ~(1 << 10)) | (__VALUE__ << 10))

void HAL_PMU_Init(void);

void HAL_PMU_EnterSleepMode(uint32_t sleepEntry);

void HAL_PMU_EnterStopMode(uint32_t stopEntry, uint32_t stopMode);

void HAL_PMU_EnterStandbyMode(void);

void HAL_PMU_EnterPowerdownMode(void);

void HAL_PMU_StandbyWakeupIOConfig(PMU_WakeUpIo_t wakeup_io, uint32_t polarity);

void HAL_PMU_StandbyWakeupIORelease(PMU_WakeUpIo_t wakeup_io);

void HAL_PMU_StandbyWakeupRTCConfig(PMU_WakeUpRtc_t wakeup_rtc);

void HAL_PMU_StandbyWakeupRTCRelease(PMU_WakeUpRtc_t wakeup_rtc);

bool HAL_PMU_ChecktStandbyStatus(void);

uint32_t HAL_PMU_GetStandbyWakeupSource(void);

void HAL_PMU_ClearStandbyWakeupStatus(uint32_t status);

void HAL_PMU_LvdResetEnable(uint32_t voltage, uint32_t filter, uint32_t filter_en);

void HAL_PMU_LvdResetDisable(void);

void HAL_PMU_BorResetEnable(uint32_t voltage);

void HAL_PMU_BorResetDisable(void);


#endif



