/*
  ******************************************************************************
  * @file    spl_adc.h
  * @version V1.0.0
  * @date    2023
  * @brief   Header file of ADC SPL module.
  ******************************************************************************
*/
#ifndef __SPL_ADC_H__
#define __SPL_ADC_H__

#include "acm32g103_spl_conf.h"



/** 
  * @brief   ADC Init structure definition  
  */ 
typedef struct
{
    uint32_t ADC_Resolution;                /*!< Configures the ADC resolution dual mode. */ 

    uint32_t ADC_ContinuousConvMode;        /*!< Specify single or continuous conversion */

    uint32_t ADC_ExternalTrigConvEdge;      /*!< Select the external trigger edge and
                                                enable the trigger of a regular group. */
    uint32_t ADC_ExternalTrigConv;          /*!< Select the external event used to trigger 
                                                the start of conversion of a regular group. */
    uint32_t ADC_DataAlign;                 /*!< Specifies whether the ADC data  alignment
                                                is left or right. */
    uint8_t  ADC_NbrOfConversion;           /*!< Specifies the number of ADC conversions
                                                that will be done using the sequencer for
                                                regular channel group.
                                                This parameter must range from 1 to 16. */                                               
}ADC_InitTypeDef;
/**
  * @}
  */ 
  
/** 
  * @brief   ADC Common Init structure definition  
  */ 
typedef struct 
{
    uint32_t ADC_Mode;                      /*!< Configures the ADC to operate in 
                                              independent or multi mode. */                                              
    uint32_t ADC_Prescaler;                 /*!< Select the frequency of the clock 
                                              to the ADC. The clock is common for all the ADCs. */
    uint32_t ADC_DMAAccessMode;             /*!< Configures the Direct memory access 
                                              mode for multi ADC mode. */
    uint32_t ADC_TwoSamplingDelay;          /*!< Configures the Delay between 2 sampling phases. */
  
}ADC_CommonInitTypeDef;
/**
  * @}
  */ 

/** 
  * @brief   ADC Common Init structure definition  
  */ 
typedef struct 
{
    uint32_t OffsetNumber;              /*!< Select the offset number
                                             Caution: Only one offset is allowed per channel. This parameter overwrites the last setting. */

    uint32_t Offset;                    /*!< Define the offset to be applied on the raw converted data.
                                            Offset value must be a positive number.
                                            Depending of ADC resolution selected (12, 10, 8 or 6 bits), this parameter must be a number between Min_Data = 0x000 and Max_Data = 0xFFF,
                                            0x3FF, 0xFF or 0x3F respectively.
                                            Note: This parameter must be modified when no conversion is on going on both regular and injected groups (ADC disabled, or ADC enabled
                                            without continuous mode or external trigger that could launch a conversion). */             

    uint32_t OffsetCalculate;           /*!< Define ADC conversion result minus or plus offset.
                                            Note: This parameter must be modified when no conversion is on going on both regular and injected groups (ADC disabled, or ADC enabled
                                            without continuous mode or external trigger that could launch a conversion). */
    uint32_t Offsetsign;                /*!< Specify whether the result format is signed or unsigned.
                                            Note: This parameter must be modified when no conversion is on going on both regular and injected groups (ADC disabled, or ADC enabled
                                            without continuous mode or external trigger that could launch a conversion). */
}ADC_OffsetConfTypeDef;
/**
  * @}
  */ 


/** @defgroup OFR1 register address offset 
  * @{
  */   
#define ADC_OFR1_ADDR_OFFSET                        (0x60)
/**
  * @}
  */

/** @defgroup JDR1 register address offset 
  * @{
  */   
#define ADC_JDR1_ADDR_OFFSET                        (0x34) 
/**
  * @}
  */

/** @defgroup ADC1/ADC2
  * @{
  */   
#define IS_ADC_INSTANCE(INSTANCE)                   (((INSTANCE) == ADC1) || ((INSTANCE) == ADC2))
/**
  * @}
  */
  
/** @defgroup ADC Trig source define
  * @{
  */ 
/*  |     Trig Source           |      Regular           |        Inject           | * 
 *  |     ADC_EXTERNAL_TIG0     |     TIM1_CC1           |      TIM1_TRGO          | *
 *  |     ADC_EXTERNAL_TIG1     |     TIM1_CC2           |      TIM1_CC4           | *
 *  |     ADC_EXTERNAL_TIG2     |     TIM1_CC3           |      TIM2_TRGO          | *
 *  |     ADC_EXTERNAL_TIG3     |     TIM2_CC2           |      TIM2_CC1           | *
 *  |     ADC_EXTERNAL_TIG4     |     TIM3_TRGO          |      TIM3_CC4           | *
 *  |     ADC_EXTERNAL_TIG5     |     TIM4_CC4           |      TIM4_TRGO          | *
 *  |     ADC_EXTERNAL_TIG6     |     EXTI Line 11       |      EXTI Line 15       | *
 *  |     ADC_EXTERNAL_TIG7     |     TIM8_TRGO          |      TIM8_CC4           | *
 *  |     ADC_EXTERNAL_TIG8     |     TIM8_TRGO2         |      TIM1_TRGO          | *
 *  |     ADC_EXTERNAL_TIG9     |     TIM1_TRGO          |      TIM8_TRGO          | *
 *  |     ADC_EXTERNAL_TIG10    |     TIM1_TRGO2         |      TIM8_TRGO2         | *
 *  |     ADC_EXTERNAL_TIG11    |     TIM2_TRGO          |      TIM3_CC3           | *
 *  |     ADC_EXTERNAL_TIG12    |     TIM4_TRGO          |      TIM3_TRGO          | *
 *  |     ADC_EXTERNAL_TIG13    |     TIM6_TRGO          |      TIM3_CC1           | *
 *  |     ADC_EXTERNAL_TIG14    |     TIM15_TRGO         |      TIM6_TRGO          | *
 *  |     ADC_EXTERNAL_TIG15    |     TIM3_CC4           |      TIM15_TRGO         | *
 *  |     ADC_EXTERNAL_TIG16    |     TIM1_CC4           |      TIM16_CC1          | *
 *  |     ADC_EXTERNAL_TIG17    |     LPTIM_OUT          |      LPTIM_OUT          | *
 *  |     ADC_EXTERNAL_TIG18    |     TIM7_TRGO          |      TIM7_TRGO          | * 
 *  |     ADC_EXTERNAL_TIG19    |     TIM8_CC4           |      TIM17_CC1          | */
#define ADC_EXTERNAL_TIG0                           (0U)
#define ADC_EXTERNAL_TIG1                           (1U)
#define ADC_EXTERNAL_TIG2                           (2U)
#define ADC_EXTERNAL_TIG3                           (3U)
#define ADC_EXTERNAL_TIG4                           (4U)
#define ADC_EXTERNAL_TIG5                           (5U)
#define ADC_EXTERNAL_TIG6                           (6U)
#define ADC_EXTERNAL_TIG7                           (7U)
#define ADC_EXTERNAL_TIG8                           (8U)
#define ADC_EXTERNAL_TIG9                           (9U)
#define ADC_EXTERNAL_TIG10                          (10U)
#define ADC_EXTERNAL_TIG11                          (11U)
#define ADC_EXTERNAL_TIG12                          (12U)
#define ADC_EXTERNAL_TIG13                          (13U)
#define ADC_EXTERNAL_TIG14                          (14U)
#define ADC_EXTERNAL_TIG15                          (15U)
#define ADC_EXTERNAL_TIG16                          (16U)
#define ADC_EXTERNAL_TIG17                          (17U)
#define ADC_EXTERNAL_TIG18                          (18U)
#define ADC_EXTERNAL_TIG19                          (19U)
#define IS_ADC_EXT_TRIG(REGTRIG)                    (((REGTRIG) == ADC_EXTERNAL_TIG0) || \
                                                    ((REGTRIG) == ADC_EXTERNAL_TIG1) || \
                                                    ((REGTRIG) == ADC_EXTERNAL_TIG2) || \
                                                    ((REGTRIG) == ADC_EXTERNAL_TIG3) || \
                                                    ((REGTRIG) == ADC_EXTERNAL_TIG4) || \
                                                    ((REGTRIG) == ADC_EXTERNAL_TIG5) || \
                                                    ((REGTRIG) == ADC_EXTERNAL_TIG6) || \
                                                    ((REGTRIG) == ADC_EXTERNAL_TIG7) || \
                                                    ((REGTRIG) == ADC_EXTERNAL_TIG8) || \
                                                    ((REGTRIG) == ADC_EXTERNAL_TIG9) || \
                                                    ((REGTRIG) == ADC_EXTERNAL_TIG10) || \
                                                    ((REGTRIG) == ADC_EXTERNAL_TIG11) || \
                                                    ((REGTRIG) == ADC_EXTERNAL_TIG12) || \
                                                    ((REGTRIG) == ADC_EXTERNAL_TIG13) || \
                                                    ((REGTRIG) == ADC_EXTERNAL_TIG14) || \
                                                    ((REGTRIG) == ADC_EXTERNAL_TIG15) || \
                                                    ((REGTRIG) == ADC_EXTERNAL_TIG16) || \
                                                    ((REGTRIG) == ADC_EXTERNAL_TIG17) || \
                                                    ((REGTRIG) == ADC_EXTERNAL_TIG18) || \
                                                    ((REGTRIG) == ADC_EXTERNAL_TIG19))  
                                                
/**
  * @}
  */                                                 
                                                
/**
  * @}
  */
   
/** @defgroup ADC External Trigger Edge 
  * @{
  */ 
#define ADC_EXTERNALTRIGCONVEDGE_NONE               (0U)
#define ADC_EXTERNALTRIGCONVEDGE_RISING             (1U)
#define ADC_EXTERNALTRIGCONVEDGE_FALLING            (2U)
#define ADC_EXTERNALTRIGCONVEDGE_RISINGFALLING      (3U)
#define IS_ADC_EXT_TRIG_EDGE(EDGE)                  (((EDGE) == ADC_EXTERNALTRIGCONVEDGE_NONE) || \
                                                    ((EDGE) == ADC_EXTERNALTRIGCONVEDGE_RISING) || \
                                                    ((EDGE) == ADC_EXTERNALTRIGCONVEDGE_FALLING) || \
                                                    ((EDGE) == ADC_EXTERNALTRIGCONVEDGE_RISINGFALLING))
/**
  * @}
  */
  
/** @defgroup ADC Data Align
* @{
*/ 
#define ADC_DATAALIGN_RIGHT                         (0x00U)
#define ADC_DATAALIGN_LEFT                          ((uint32_t)ADC_CR2_ALIGN)
#define IS_ADC_DATA_ALIGN(ALIGN)                    (((ALIGN) == ADC_DATAALIGN_RIGHT) || ((ALIGN) == ADC_DATAALIGN_LEFT))
/**
  * @}
  */ 

/** @defgroup ADC Continuous
* @{
*/ 
#define ADC_CONT_DISABLE                            (0x00U)
#define ADC_CONT_ENABLE                             ((uint32_t)ADC_CR1_CONT)
#define IS_ADC_CONT(CONT)                           (((CONT) == ADC_CONT_DISABLE) || ((CONT) == ADC_CONT_ENABLE))
/**
  * @}
  */    
  
/** @defgroup ADC Resolution
  * @{
  */
#define ADC_RESOLUTION_12B                          (0x00U)   /*!< ADC resolution 12 bits */
#define ADC_RESOLUTION_10B                          (ADC_CR2_RES_0)   /*!< ADC resolution 10 bits */
#define ADC_RESOLUTION_8B                           (ADC_CR2_RES_1)   /*!< ADC resolution  8 bits */
#define ADC_RESOLUTION_6B                           (ADC_CR2_RES_0 | ADC_CR2_RES_1)   /*!< ADC resolution  6 bits */
#define IS_ADC_RESOLUTION(RESOLUTION)               (((RESOLUTION) == ADC_RESOLUTION_12B) || \
                                                    ((RESOLUTION) == ADC_RESOLUTION_10B) || \
                                                    ((RESOLUTION) == ADC_RESOLUTION_8B) || \
                                                    ((RESOLUTION) == ADC_RESOLUTION_6B))
/**
  * @}
  */
    
/** @defgroup ADC ClockPrescale 
  * @{
  */ 
#define ADC_CLOCK_DIV1                              (0x00U)    
#define ADC_CLOCK_DIV2                              (ADC_CCR_ADCDIV_0)    
#define ADC_CLOCK_DIV3                              (ADC_CCR_ADCDIV_1)    
#define ADC_CLOCK_DIV4                              (ADC_CCR_ADCDIV_0 | ADC_CCR_ADCDIV_1)    
#define ADC_CLOCK_DIV5                              (ADC_CCR_ADCDIV_2)    
#define ADC_CLOCK_DIV6                              (ADC_CCR_ADCDIV_0 | ADC_CCR_ADCDIV_2)    
#define ADC_CLOCK_DIV7                              (ADC_CCR_ADCDIV_1 | ADC_CCR_ADCDIV_2)    
#define ADC_CLOCK_DIV8                              (ADC_CCR_ADCDIV_0 | ADC_CCR_ADCDIV_1 | ADC_CCR_ADCDIV_2)    
#define ADC_CLOCK_DIV9                              (ADC_CCR_ADCDIV_3)    
#define ADC_CLOCK_DIV10                             (ADC_CCR_ADCDIV_0 | ADC_CCR_ADCDIV_3)    
#define ADC_CLOCK_DIV11                             (ADC_CCR_ADCDIV_1 | ADC_CCR_ADCDIV_3)    
#define ADC_CLOCK_DIV12                             (ADC_CCR_ADCDIV_0 | ADC_CCR_ADCDIV_1 | ADC_CCR_ADCDIV_3)    
#define ADC_CLOCK_DIV13                             (ADC_CCR_ADCDIV_2 | ADC_CCR_ADCDIV_3)    
#define ADC_CLOCK_DIV14                             (ADC_CCR_ADCDIV_0 | ADC_CCR_ADCDIV_2 | ADC_CCR_ADCDIV_3)    
#define ADC_CLOCK_DIV15                             (ADC_CCR_ADCDIV_1 | ADC_CCR_ADCDIV_2 | ADC_CCR_ADCDIV_3)    
#define ADC_CLOCK_DIV16                             (ADC_CCR_ADCDIV_0 | ADC_CCR_ADCDIV_1 | ADC_CCR_ADCDIV_2 | ADC_CCR_ADCDIV_3) 
#define IS_ADC_CLOCKDIV(CLOCKDIV)                   (((CLOCKDIV) == ADC_CLOCK_DIV1) || \
                                                    ((CLOCKDIV) == ADC_CLOCK_DIV2) || \
                                                    ((CLOCKDIV) == ADC_CLOCK_DIV3) || \
                                                    ((CLOCKDIV) == ADC_CLOCK_DIV4) || \
                                                    ((CLOCKDIV) == ADC_CLOCK_DIV5) || \
                                                    ((CLOCKDIV) == ADC_CLOCK_DIV6) || \
                                                    ((CLOCKDIV) == ADC_CLOCK_DIV7) || \
                                                    ((CLOCKDIV) == ADC_CLOCK_DIV8) || \
                                                    ((CLOCKDIV) == ADC_CLOCK_DIV9) || \
                                                    ((CLOCKDIV) == ADC_CLOCK_DIV10) || \
                                                    ((CLOCKDIV) == ADC_CLOCK_DIV11) || \
                                                    ((CLOCKDIV) == ADC_CLOCK_DIV12) || \
                                                    ((CLOCKDIV) == ADC_CLOCK_DIV13) || \
                                                    ((CLOCKDIV) == ADC_CLOCK_DIV14) || \
                                                    ((CLOCKDIV) == ADC_CLOCK_DIV15) || \
                                                    ((CLOCKDIV) == ADC_CLOCK_DIV16))
/**
  * @}
  */
  
/** @defgroup  Multimode
  * @{
  */
#define ADC_MODE_INDEPENDENT                        (0x00U)                                          /*!< ADC dual mode disabled (ADC independent mode) */
#define ADC_DUALMODE_REGSIMULT_INJECSIMULT          (ADC_CCR_DUALMOD_0) /*!< ADC dual mode enabled: Combined group regular simultaneous + group injected simultaneous */
#define ADC_DUALMODE_REGSIMULT_ALTERTRIG            (ADC_CCR_DUALMOD_1) /*!< ADC dual mode enabled: Combined group regular simultaneous + group injected alternate trigger */
#define ADC_DUALMODE_REGINTERL_INJECSIMULT          (ADC_CCR_DUALMOD_0 | ADC_CCR_DUALMOD_1) /*!< ADC dual mode enabled: Combined group regular interleaved + group injected simultaneous */
#define ADC_DUALMODE_INJECSIMULT                    (ADC_CCR_DUALMOD_0 | ADC_CCR_DUALMOD_2) /*!< ADC dual mode enabled: group injected simultaneous */
#define ADC_DUALMODE_REGSIMULT                      (ADC_CCR_DUALMOD_1 | ADC_CCR_DUALMOD_2) /*!< ADC dual mode enabled: group regular simultaneous */
#define ADC_DUALMODE_INTERL                         (ADC_CCR_DUALMOD_0 | ADC_CCR_DUALMOD_1 | ADC_CCR_DUALMOD_2) /*!< ADC dual mode enabled: Combined group regular interleaved */
#define ADC_DUALMODE_ALTERTRIG                      (ADC_CCR_DUALMOD_0 | ADC_CCR_DUALMOD_3) /*!< ADC dual mode enabled: group injected alternate trigger. Works only with external triggers (not internal SW start) */
#define IS_ADC_MODE(MODE)                           (((MODE) == ADC_MODE_INDEPENDENT) || \
                                                    ((MODE) == ADC_DUALMODE_REGSIMULT_INJECSIMULT) || \
                                                    ((MODE) == ADC_DUALMODE_REGSIMULT_ALTERTRIG) || \
                                                    ((MODE) == ADC_DUALMODE_REGINTERL_INJECSIMULT) || \
                                                    ((MODE) == ADC_DUALMODE_INJECSIMULT) || \
                                                    ((MODE) == ADC_DUALMODE_REGSIMULT) || \
                                                    ((MODE) == ADC_DUALMODE_INTERL) || \
                                                    ((MODE) == ADC_DUALMODE_ALTERTRIG))
/**
  * @}
  */
  
/** @defgroup  DMA transfer mode depending on ADC resolution
  * @{
  */
#define ADC_DMAACCESSMODE_DISABLED                  (0x00U)     /*!< DMA multimode disabled: each ADC uses its own DMA channel */
#define ADC_DMAACCESSMODE_12_10_BITS                (ADC_CCR_DMADUAL_1)   /*!< DMA multimode enabled (one DMA channel for both ADC, DMA of ADC master) for 12 and 10 bits resolution */
#define ADC_DMAACCESSMODE_8_6_BITS                  (ADC_CCR_DMADUAL_0 | ADC_CCR_DMADUAL_1)     /*!< DMA multimode enabled (one DMA channel for both ADC, DMA of ADC master) for 8 and 6 bits resolution */
#define IS_ADC_DMA_ACCESS_MODE(MODE)                (((MODE) == ADC_DMAACCESSMODE_DISABLED) || \
                                                    ((MODE) == ADC_DMAACCESSMODE_12_10_BITS) || \
                                                    ((MODE) == ADC_DMAACCESSMODE_8_6_BITS))
/**
  * @}
  */
  
/** @defgroup  ADC Delay Between 2 Sampling Phases
  * @{
  */ 
#define ADC_TWOSAMPLINGDELAY_5CYCLES                (0x00U)
#define ADC_TWOSAMPLINGDELAY_6CYCLES                (ADC_CCR_DELAY_0)
#define ADC_TWOSAMPLINGDELAY_7CYCLES                (ADC_CCR_DELAY_1)
#define ADC_TWOSAMPLINGDELAY_8CYCLES                (ADC_CCR_DELAY_0 | ADC_CCR_DELAY_1)
#define ADC_TWOSAMPLINGDELAY_9CYCLES                (ADC_CCR_DELAY_2)
#define ADC_TWOSAMPLINGDELAY_10CYCLES               (ADC_CCR_DELAY_0 | ADC_CCR_DELAY_2)
#define ADC_TWOSAMPLINGDELAY_11CYCLES               (ADC_CCR_DELAY_1 | ADC_CCR_DELAY_2)
#define ADC_TWOSAMPLINGDELAY_12CYCLES               (ADC_CCR_DELAY_0 | ADC_CCR_DELAY_1 | ADC_CCR_DELAY_2)
#define ADC_TWOSAMPLINGDELAY_13CYCLES               (ADC_CCR_DELAY_3)
#define ADC_TWOSAMPLINGDELAY_14CYCLES               (ADC_CCR_DELAY_0 | ADC_CCR_DELAY_3)
#define ADC_TWOSAMPLINGDELAY_15CYCLES               (ADC_CCR_DELAY_1 | ADC_CCR_DELAY_3)
#define ADC_TWOSAMPLINGDELAY_16CYCLES               (ADC_CCR_DELAY_0 | ADC_CCR_DELAY_1 | ADC_CCR_DELAY_3)
#define ADC_TWOSAMPLINGDELAY_17CYCLES               (ADC_CCR_DELAY_2 | ADC_CCR_DELAY_3)
#define ADC_TWOSAMPLINGDELAY_18CYCLES               (ADC_CCR_DELAY_0 | ADC_CCR_DELAY_2 | ADC_CCR_DELAY_3)
#define ADC_TWOSAMPLINGDELAY_19CYCLES               (ADC_CCR_DELAY_1 | ADC_CCR_DELAY_2 | ADC_CCR_DELAY_3)
#define ADC_TWOSAMPLINGDELAY_20CYCLES               (ADC_CCR_DELAY_0 | ADC_CCR_DELAY_1 | ADC_CCR_DELAY_2 | ADC_CCR_DELAY_3)                                    
#define IS_ADC_SAMPLING_DELAY(DELAY)                (((DELAY) == ADC_TWOSAMPLINGDELAY_5CYCLES) || \
                                                    ((DELAY) == ADC_TWOSAMPLINGDELAY_6CYCLES) || \
                                                    ((DELAY) == ADC_TWOSAMPLINGDELAY_7CYCLES) || \
                                                    ((DELAY) == ADC_TWOSAMPLINGDELAY_8CYCLES) || \
                                                    ((DELAY) == ADC_TWOSAMPLINGDELAY_9CYCLES) || \
                                                    ((DELAY) == ADC_TWOSAMPLINGDELAY_10CYCLES) || \
                                                    ((DELAY) == ADC_TWOSAMPLINGDELAY_11CYCLES) || \
                                                    ((DELAY) == ADC_TWOSAMPLINGDELAY_12CYCLES) || \
                                                    ((DELAY) == ADC_TWOSAMPLINGDELAY_13CYCLES) || \
                                                    ((DELAY) == ADC_TWOSAMPLINGDELAY_14CYCLES) || \
                                                    ((DELAY) == ADC_TWOSAMPLINGDELAY_15CYCLES) || \
                                                    ((DELAY) == ADC_TWOSAMPLINGDELAY_16CYCLES) || \
                                                    ((DELAY) == ADC_TWOSAMPLINGDELAY_17CYCLES) || \
                                                    ((DELAY) == ADC_TWOSAMPLINGDELAY_18CYCLES) || \
                                                    ((DELAY) == ADC_TWOSAMPLINGDELAY_19CYCLES) || \
                                                    ((DELAY) == ADC_TWOSAMPLINGDELAY_20CYCLES))

/**
  * @}
  */


/** @defgroup ADC analog watchdog selection
  * @{
  */ 
#define ADC_ANALOGWATCHDOG_RCH_SINGLE               (ADC_CR1_AWDSGL | ADC_CR1_AWDEN)   //Single regular channel
#define ADC_ANALOGWATCHDOG_JCH_SINGLE               (ADC_CR1_AWDSGL | ADC_CR1_JAWDEN)   //Single Inject channel
#define ADC_ANALOGWATCHDOG_RCH_OR_JCH_SINGLE        (ADC_CR1_AWDSGL | ADC_CR1_AWDEN | ADC_CR1_JAWDEN)   //Regular or inject channel
#define ADC_ANALOGWATCHDOG_RCH_ALL                  (ADC_CR1_AWDEN)   //All regular channels
#define ADC_ANALOGWATCHDOG_JCH_ALL                  (ADC_CR1_JAWDEN)   //All inject channels
#define ADC_ANALOGWATCHDOG_RCH_AND_JCH_ALL          (ADC_CR1_AWDEN | ADC_CR1_JAWDEN)   //All regular and inject channels
#define ADC_ANALOGWATCHDOG_NONE                     (0x00U)
#define IS_ADC_ANALOG_WATCHDOG(WATCHDOG)            (((WATCHDOG) == ADC_ANALOGWATCHDOG_RCH_SINGLE) || \
                                                    ((WATCHDOG) == ADC_ANALOGWATCHDOG_JCH_SINGLE) || \
                                                    ((WATCHDOG) == ADC_ANALOGWATCHDOG_RCH_OR_JCH_SINGLE) || \
                                                    ((WATCHDOG) == ADC_ANALOGWATCHDOG_RCH_ALL) || \
                                                    ((WATCHDOG) == ADC_ANALOGWATCHDOG_JCH_ALL) || \
                                                    ((WATCHDOG) == ADC_ANALOGWATCHDOG_RCH_AND_JCH_ALL) || \
                                                    ((WATCHDOG) == ADC_ANALOGWATCHDOG_NONE))  
/**
  * @}
  */
  
/** @defgroup ADC analog watchdog thresholds 
  * @{
  */ 
#define IS_ADC_THRESHOLD(THRESHOLD) ((THRESHOLD) <= 0xFFF)
/**
  * @}
  */   
  
/** @defgroup ADC channel number 
* @{
*/
#define ADC_CHANNEL_0                               (0U)    
#define ADC_CHANNEL_1                               (1U)    
#define ADC_CHANNEL_2                               (2U)    
#define ADC_CHANNEL_3                               (3U)    
#define ADC_CHANNEL_4                               (4U)    
#define ADC_CHANNEL_5                               (5U)    
#define ADC_CHANNEL_6                               (6U)    
#define ADC_CHANNEL_7                               (7U)    
#define ADC_CHANNEL_8                               (8U)    
#define ADC_CHANNEL_9                               (9U)    
#define ADC_CHANNEL_10                              (10U)    
#define ADC_CHANNEL_11                              (11U)    
#define ADC_CHANNEL_12                              (12U)    
#define ADC_CHANNEL_13                              (13U)    
#define ADC_CHANNEL_14                              (14U)    
#define ADC_CHANNEL_15                              (15U)    
#define ADC_CHANNEL_16                              (16U) 
#define ADC_CHANNEL_17                              (17U)          
#define ADC_CHANNEL_18                              (18U)    
#define ADC_CHANNEL_19                              (19U)    
#define IS_ADC_CHANNEL(CHANNEL)                     (((CHANNEL) == ADC_CHANNEL_0) || \
                                                    ((CHANNEL) == ADC_CHANNEL_1) || \
                                                    ((CHANNEL) == ADC_CHANNEL_2) || \
                                                    ((CHANNEL) == ADC_CHANNEL_3) || \
                                                    ((CHANNEL) == ADC_CHANNEL_4) || \
                                                    ((CHANNEL) == ADC_CHANNEL_5) || \
                                                    ((CHANNEL) == ADC_CHANNEL_6) || \
                                                    ((CHANNEL) == ADC_CHANNEL_7) || \
                                                    ((CHANNEL) == ADC_CHANNEL_8) || \
                                                    ((CHANNEL) == ADC_CHANNEL_9) || \
                                                    ((CHANNEL) == ADC_CHANNEL_10) || \
                                                    ((CHANNEL) == ADC_CHANNEL_11) || \
                                                    ((CHANNEL) == ADC_CHANNEL_12) || \
                                                    ((CHANNEL) == ADC_CHANNEL_13) || \
                                                    ((CHANNEL) == ADC_CHANNEL_14) || \
                                                    ((CHANNEL) == ADC_CHANNEL_15) || \
                                                    ((CHANNEL) == ADC_CHANNEL_16) || \
                                                    ((CHANNEL) == ADC_CHANNEL_17) || \
                                                    ((CHANNEL) == ADC_CHANNEL_18) || \
                                                    ((CHANNEL) == ADC_CHANNEL_19))
/**
  * @}
  */ 

/** @defgroup Sampling time
  * @{
  */
#define ADC_SMP_CLOCK_1                             (0U)
#define ADC_SMP_CLOCK_3                             (1U)
#define ADC_SMP_CLOCK_5                             (2U)
#define ADC_SMP_CLOCK_7                             (3U)
#define ADC_SMP_CLOCK_10                            (4U)
#define ADC_SMP_CLOCK_13                            (5U)
#define ADC_SMP_CLOCK_16                            (6U)
#define ADC_SMP_CLOCK_20                            (7U)
#define ADC_SMP_CLOCK_30                            (8U)
#define ADC_SMP_CLOCK_60                            (9U)
#define ADC_SMP_CLOCK_80                            (10U)
#define ADC_SMP_CLOCK_100                           (11U)
#define ADC_SMP_CLOCK_120                           (12U)
#define ADC_SMP_CLOCK_160                           (13U)
#define ADC_SMP_CLOCK_320                           (14U)
#define ADC_SMP_CLOCK_480                           (15U)
#define IS_ADC_SMPCLOCK(SMPCLOCK)                   (((SMPCLOCK) == ADC_SMP_CLOCK_1) || \
                                                    ((SMPCLOCK) == ADC_SMP_CLOCK_3) || \
                                                    ((SMPCLOCK) == ADC_SMP_CLOCK_5) || \
                                                    ((SMPCLOCK) == ADC_SMP_CLOCK_7) || \
                                                    ((SMPCLOCK) == ADC_SMP_CLOCK_10) || \
                                                    ((SMPCLOCK) == ADC_SMP_CLOCK_13) || \
                                                    ((SMPCLOCK) == ADC_SMP_CLOCK_16) || \
                                                    ((SMPCLOCK) == ADC_SMP_CLOCK_20) || \
                                                    ((SMPCLOCK) == ADC_SMP_CLOCK_30) || \
                                                    ((SMPCLOCK) == ADC_SMP_CLOCK_60) || \
                                                    ((SMPCLOCK) == ADC_SMP_CLOCK_80) || \
                                                    ((SMPCLOCK) == ADC_SMP_CLOCK_100) || \
                                                    ((SMPCLOCK) == ADC_SMP_CLOCK_120) || \
                                                    ((SMPCLOCK) == ADC_SMP_CLOCK_160) || \
                                                    ((SMPCLOCK) == ADC_SMP_CLOCK_320) || \
                                                    ((SMPCLOCK) == ADC_SMP_CLOCK_480))   
/**
  * @}
  */ 
  
/** @defgroup ADC sequence number
* @{
*/
#define ADC_SEQUENCE_SQ1                            (1U)
#define ADC_SEQUENCE_SQ2                            (2U)
#define ADC_SEQUENCE_SQ3                            (3U)
#define ADC_SEQUENCE_SQ4                            (4U)
#define ADC_SEQUENCE_SQ5                            (5U)
#define ADC_SEQUENCE_SQ6                            (6U)
#define ADC_SEQUENCE_SQ7                            (7U)
#define ADC_SEQUENCE_SQ8                            (8U)
#define ADC_SEQUENCE_SQ9                            (9U)
#define ADC_SEQUENCE_SQ10                           (10U)
#define ADC_SEQUENCE_SQ11                           (11U)
#define ADC_SEQUENCE_SQ12                           (12U)
#define ADC_SEQUENCE_SQ13                           (13U)
#define ADC_SEQUENCE_SQ14                           (14U)
#define ADC_SEQUENCE_SQ15                           (15U)
#define ADC_SEQUENCE_SQ16                           (16U)
#define IS_ADC_SEQUENCE(SEQUENCE)                   (((SEQUENCE) == ADC_SEQUENCE_SQ1) || \
                                                    ((SEQUENCE) == ADC_SEQUENCE_SQ2) || \
                                                    ((SEQUENCE) == ADC_SEQUENCE_SQ3) || \
                                                    ((SEQUENCE) == ADC_SEQUENCE_SQ4) || \
                                                    ((SEQUENCE) == ADC_SEQUENCE_SQ5) || \
                                                    ((SEQUENCE) == ADC_SEQUENCE_SQ6) || \
                                                    ((SEQUENCE) == ADC_SEQUENCE_SQ7) || \
                                                    ((SEQUENCE) == ADC_SEQUENCE_SQ8) || \
                                                    ((SEQUENCE) == ADC_SEQUENCE_SQ9) || \
                                                    ((SEQUENCE) == ADC_SEQUENCE_SQ10) || \
                                                    ((SEQUENCE) == ADC_SEQUENCE_SQ11) || \
                                                    ((SEQUENCE) == ADC_SEQUENCE_SQ12) || \
                                                    ((SEQUENCE) == ADC_SEQUENCE_SQ13) || \
                                                    ((SEQUENCE) == ADC_SEQUENCE_SQ14) || \
                                                    ((SEQUENCE) == ADC_SEQUENCE_SQ15) || \
                                                    ((SEQUENCE) == ADC_SEQUENCE_SQ16))
/**
  * @}
  */   

/** @defgroup ADC regular length 
  * @{
  */ 
#define IS_ADC_REGULAR_LENGTH(LENGTH)               (((LENGTH) >= 0x01) && ((LENGTH) <= 0x10))
/**
  * @}
  */ 
  
/** @defgroup ADC regular discontinuous mode number 
  * @{
  */ 
#define IS_ADC_REGULAR_DISC_NUMBER(NUMBER)          (((NUMBER) >= 0x1) && ((NUMBER) <= 0x8))
/**
  * @}
  */ 

/** @defgroup ADC injected rank 
  * @{
  */ 
#define ADC_INJECTED_RANK_1                         (1U)
#define ADC_INJECTED_RANK_2                         (2U)
#define ADC_INJECTED_RANK_3                         (3U)
#define ADC_INJECTED_RANK_4                         (4U)
#define IS_ADC_INJECTED_RANK(RANK)                  (((RANK) >= 0x1) && ((RANK) <= 0x4))
/**
  * @}
  */ 

/** @defgroup ADC injected length 
  * @{
  */ 
#define IS_ADC_INJECTED_LENGTH(LENGTH)              (((LENGTH) >= 0x1) && ((LENGTH) <= 0x4))
/**
  * @}
  */  
  
/** @defgroup Offset Number 
  * @{
  */ 
#define ADC_OFR_NONE                                (0U) /*!< ADC offset disabled: no offset correction for the selected ADC channel */
#define ADC_OFR_1                                   (1U) /*!< ADC offset number 1: ADC_OFR1 */
#define ADC_OFR_2                                   (2U) /*!< ADC offset number 2: ADC_OFR2 */
#define ADC_OFR_3                                   (3U) /*!< ADC offset number 3: ADC_OFR3 */
#define ADC_OFR_4                                   (4U) /*!< ADC offset number 4: ADC_OFR4 */
#define IS_ADC_OFFSET_NUMBER(NUMBER)                (((NUMBER) >= 0x1) && ((NUMBER) <= 0x4))
/**
  * @}
  */ 
  
/** @defgroup offset 
  * @{
  */ 
#define IS_ADC_OFFSET(OFFSET)                       ((OFFSET) <= 0xFFF)
/**
  * @}
  */ 

 /** @defgroup Offset Calculate
  * @{
  */
#define ADC_OFFSET_MINUS                            (0x00000000UL)  /*!< ADC conversion result minus offset */
#define ADC_OFFSET_PLUS                             (ADC_OFRX_OFFSETXPOS)   /*!< ADC conversion result plus offset */
#define IS_ADC_OFFSET_CALCULATE(CALCULATE)          (((CALCULATE) == ADC_OFFSET_MINUS) || ((CALCULATE) == ADC_OFFSET_PLUS))
/**
  * @}
  */
  

/** @defgroup Offset sign
  * @{
  */
#define ADC_OFFSET_SIGN_SIGNED                      (0x00000000UL)                  /*!< ADC offset saturation is disabled (among ADC selected offset number 1, 2, 3 or 4) */
#define ADC_OFFSET_SIGN_UNSIGNED                    (ADC_OFRX_OFFSETXSAT)           /*!< ADC offset saturation is enabled (among ADC selected offset number 1, 2, 3 or 4) */
#define IS_ADC_OFFSET_SIGN(SIGN)                    (((SIGN) == ADC_OFFSET_SIGN_SIGNED) || ((SIGN) == ADC_OFFSET_SIGN_UNSIGNED))
/**
  * @}
  */ 

/** @defgroup JDR Number 
  * @{
  */ 
#define ADC_JDR_NONE                                (0U) /*!< none */
#define ADC_JDR_1                                   (1U) /*!< ADC JDR number 1: ADC_JDR1 */
#define ADC_JDR_2                                   (2U) /*!< ADC JDR number 2: ADC_JDR2 */
#define ADC_JDR_3                                   (3U) /*!< ADC JDR number 3: ADC_JDR3 */
#define ADC_JDR_4                                   (4U) /*!< ADC JDR number 4: ADC_JDR4 */
#define IS_ADC_JDR_NUMBER(NUMBER)                   (((NUMBER) >= 0x1) && ((NUMBER) <= 0x4))
/**
  * @}
  */ 

/** @defgroup  Oversampling Data shift
  * @{
  */
#define ADC_RIGHTBITSHIFT_NONE                      (0U) /*!< ADC oversampling no shift (sum of the ADC conversions data is not divided */
#define ADC_RIGHTBITSHIFT_1                         (1U) /*!< ADC oversampling shift of 1 (sum of the ADC conversions data is divided by 2 */
#define ADC_RIGHTBITSHIFT_2                         (2U) /*!< ADC oversampling shift of 2 (sum of the ADC conversions data is divided by 4 */
#define ADC_RIGHTBITSHIFT_3                         (3U) /*!< ADC oversampling shift of 3 (sum of the ADC conversions data is divided by 8 */
#define ADC_RIGHTBITSHIFT_4                         (4U) /*!< ADC oversampling shift of 4 (sum of the ADC conversions data is divided by 16 */
#define ADC_RIGHTBITSHIFT_5                         (5U) /*!< ADC oversampling shift of 5 (sum of the ADC conversions data is divided by 32 */
#define ADC_RIGHTBITSHIFT_6                         (6U) /*!< ADC oversampling shift of 6 (sum of the ADC conversions data is divided by 64 */
#define ADC_RIGHTBITSHIFT_7                         (7U) /*!< ADC oversampling shift of 7 (sum of the ADC conversions data is divided by 128 */
#define ADC_RIGHTBITSHIFT_8                         (8U) /*!< ADC oversampling shift of 8 (sum of the ADC conversions data is divided by 256 */
#define IS_ADC_RIGHTBITSHIFT(SHIFT)                 (((SHIFT) == ADC_RIGHTBITSHIFT_NONE) || \
                                                    ((SHIFT) == ADC_RIGHTBITSHIFT_1) || \
                                                    ((SHIFT) == ADC_RIGHTBITSHIFT_2) || \
                                                    ((SHIFT) == ADC_RIGHTBITSHIFT_3) || \
                                                    ((SHIFT) == ADC_RIGHTBITSHIFT_4) || \
                                                    ((SHIFT) == ADC_RIGHTBITSHIFT_5) || \
                                                    ((SHIFT) == ADC_RIGHTBITSHIFT_6) || \
                                                    ((SHIFT) == ADC_RIGHTBITSHIFT_7) || \
                                                    ((SHIFT) == ADC_RIGHTBITSHIFT_8))
/**
  * @}
  */
  
/** @defgroup  Oversampling Ratio
  * @{
  */
#define ADC_OVERSAMPLING_RATIO_2                    (0U)  /*!< ADC oversampling ratio of 2 (2 ADC conversions are performed */
#define ADC_OVERSAMPLING_RATIO_4                    (1U)  /*!< ADC oversampling ratio of 4 (4 ADC conversions are performed */
#define ADC_OVERSAMPLING_RATIO_8                    (2U)  /*!< ADC oversampling ratio of 8 (8 ADC conversions are performed */
#define ADC_OVERSAMPLING_RATIO_16                   (3U)  /*!< ADC oversampling ratio of 16 (16 ADC conversions are performed */
#define ADC_OVERSAMPLING_RATIO_32                   (4U)  /*!< ADC oversampling ratio of 32 (32 ADC conversions are performed */
#define ADC_OVERSAMPLING_RATIO_64                   (5U)  /*!< ADC oversampling ratio of 64 (64 ADC conversions are performed */
#define ADC_OVERSAMPLING_RATIO_128                  (6U)  /*!< ADC oversampling ratio of 128 (128 ADC conversions are performed */
#define ADC_OVERSAMPLING_RATIO_256                  (7U)  /*!< ADC oversampling ratio of 256 (256 ADC conversions are performed */
#define IS_ADC_OVERSAMPLING_RATIO(RATIO)            (((RATIO) == ADC_OVERSAMPLING_RATIO_2) || \
                                                    ((RATIO) == ADC_OVERSAMPLING_RATIO_4)  || \
                                                    ((RATIO) == ADC_OVERSAMPLING_RATIO_8)  || \
                                                    ((RATIO) == ADC_OVERSAMPLING_RATIO_16) || \
                                                    ((RATIO) == ADC_OVERSAMPLING_RATIO_32) || \
                                                    ((RATIO) == ADC_OVERSAMPLING_RATIO_64) || \
                                                    ((RATIO) == ADC_OVERSAMPLING_RATIO_128) || \
                                                    ((RATIO) == ADC_OVERSAMPLING_RATIO_256))
/**
  * @}
  */  

/** @defgroup overflow mode
  * @{
  */
#define ADC_OVERFLOWMODE_LAST                       (0x00000000UL)  /*!< ADC keeps the last sampling data when overflow occurs */
#define ADC_OVERFLOWMODE_NEW                        (ADC_CR2_OVRMOD) /*!< ADC saves the new sampling data when overflow occurs */
#define IS_ADC_OVERFLOW_MODE(MODE)                  (((MODE) == ADC_OVERFLOWMODE_LAST) || ((MODE) == ADC_OVERFLOWMODE_NEW))
/**
  * @}
  */
  
/** @defgroup oversampling triger mode
  * @{
  */
#define ADC_OVERSAMPLING_TRIGER_MORE                (0x00000000UL)  /*!< One trigger for N ADC conversion */
#define ADC_OVERSAMPLING_TRIGER_ONCE                (ADC_CR2_TROVS) /*!< One trigger for ADC conversion */
#define IS_ADC_OVERSAMPLING_TRIGER(MODE)            (((MODE) == ADC_OVERSAMPLING_TRIGER_MORE) || ((MODE) == ADC_OVERSAMPLING_TRIGER_ONCE))
/**
  * @}
  */

/** @defgroup ADC interrupts definition 
  * @{
  */ 
#define ADC_IT_FLAG_EOSMP                           (0x00000002UL)  
#define ADC_IT_FLAG_EOC                             (0x00000004UL)  
#define ADC_IT_FLAG_EOG                             (0x00000008UL)  
#define ADC_IT_FLAG_OVERF                           (0x00000010UL) 
#define ADC_IT_FLAG_JEOC                            (0x00000020UL) 
#define ADC_IT_FLAG_JEOG                            (0x00000040UL) 
#define ADC_IT_FLAG_AWD                             (0x00000080UL)
#define ADC_IT_FLAG_ALL                             (0x000000FFUL)    
#define IS_ADC_IT_FLAG(FLAG)                        (((FLAG) == ADC_IT_FLAG_EOSMP) || ((FLAG) == ADC_IT_FLAG_EOC) || \
                                                    ((FLAG) == ADC_IT_FLAG_EOG)|| ((FLAG) == ADC_IT_FLAG_OVERF) || \
                                                    ((FLAG) == ADC_IT_FLAG_JEOC)|| ((FLAG) == ADC_IT_FLAG_JEOG) || \
                                                    ((FLAG) == ADC_IT_FLAG_AWD) || ((FLAG) == ADC_IT_FLAG_ALL))   
/**
  * @}
  */ 
  
/** @defgroup Single-ended or differential 
* @{
*/
#define ADC_DIFF_SINGLE                             (0U)   /*!< ADC channel ending set to single ended */
#define ADC_DIFF_DIFFERENTIAL                       (1U)   /*!< ADC channel ending set to differential */  
#define IS_ADC_DIFF(MODE)                           (((MODE) == ADC_DIFF_SINGLE) || ((MODE) == ADC_DIFF_DIFFERENTIAL))
/**
  * @}
  */ 
  
  /** @defgroup DIFF or SIGN channel number 
* @{
*/
#define ADC_CHANNEL_0_10                            (0U)    
#define ADC_CHANNEL_1_11                            (1U)    
#define ADC_CHANNEL_2_12                            (2U)    
#define ADC_CHANNEL_3_13                            (3U)    
#define ADC_CHANNEL_4_14                            (4U)    
#define ADC_CHANNEL_5_15                            (5U)    
#define ADC_CHANNEL_6_16                            (6U)    
#define ADC_CHANNEL_7_17                            (7U)    
#define ADC_CHANNEL_8_18                            (8U)    
#define ADC_CHANNEL_9_19                            (9U)       
#define IS_ADC_DIFF_SIGN_CHANNEL(CHANNEL)           (((CHANNEL) == ADC_CHANNEL_0_10) || \
                                                    ((CHANNEL) == ADC_CHANNEL_1_11) || \
                                                    ((CHANNEL) == ADC_CHANNEL_2_12) || \
                                                    ((CHANNEL) == ADC_CHANNEL_3_13) || \
                                                    ((CHANNEL) == ADC_CHANNEL_4_14) || \
                                                    ((CHANNEL) == ADC_CHANNEL_5_15) || \
                                                    ((CHANNEL) == ADC_CHANNEL_6_16) || \
                                                    ((CHANNEL) == ADC_CHANNEL_7_17) || \
                                                    ((CHANNEL) == ADC_CHANNEL_8_18) || \
                                                    ((CHANNEL) == ADC_CHANNEL_9_19))
/**
  * @}
  */ 

/** @defgroup Signed or unsigned number selection
* @{
*/
#define ADC_SIGN_UNSIGNED                           (0U)   /*!< The result is an unsigned number */
#define ADC_SIGN_SIGNED                             (1U)   /*!< The result is an signed number */  
#define IS_ADC_SIGN(MODE)                           (((MODE) == ADC_SIGN_SIGNED) || ((MODE) == ADC_SIGN_UNSIGNED))
/**
  * @}
  */ 




/*  Function used to set the ADC configuration to the default reset state */
void ADC_DeInit(void);

/* Initialization and Configuration functions */
void ADC_Init(ADC_TypeDef* ADCx, ADC_InitTypeDef* ADC_InitStruct);
void ADC_StructInit(ADC_InitTypeDef* ADC_InitStruct);
void ADC_CommonInit(ADC_CommonInitTypeDef* ADC_CommonInitStruct);
void ADC_CommonStructInit(ADC_CommonInitTypeDef* ADC_CommonInitStruct);
void ADC_Cmd(ADC_TypeDef* ADCx, FunctionalState NewState);
void ADC_FastModeCmd(ADC_TypeDef* ADCx, FunctionalState NewState);
void ADC_StopControlCmd(ADC_TypeDef* ADCx, FunctionalState NewState);
void ADC_OverflowmodeConfig(ADC_TypeDef* ADCx, uint32_t Overflowmode);
void ADC_DifferentialConfig(ADC_TypeDef* ADCx, uint8_t Channel, uint8_t Differential);
void ADC_SignConfig(ADC_TypeDef* ADCx, uint8_t Channel, uint8_t Sign);
void ADC_OffsetConfig(ADC_TypeDef* ADCx, uint8_t ADC_Channel, ADC_OffsetConfTypeDef *Offset_InitStruct);
void ADC_OverSamplingConfig(ADC_TypeDef* ADCx, uint8_t Ratio, uint8_t Shift);
void ADC_OverSamplingCmd(ADC_TypeDef* ADCx, FunctionalState NewState);
void ADC_OverSamplingTrigerConfig(ADC_TypeDef* ADCx, uint32_t TrigerMode);
void ADC_InjectOverSamplingCmd(ADC_TypeDef* ADCx, FunctionalState NewState);

/* Analog Watchdog configuration functions */
void ADC_AnalogWatchdogCmd(ADC_TypeDef* ADCx, uint32_t ADC_AnalogWatchdog);
void ADC_AnalogWatchdogThresholdsConfig(ADC_TypeDef* ADCx, uint16_t HighThreshold, uint16_t LowThreshold);
void ADC_AnalogWatchdogThresholdsOfDifferConfig(ADC_TypeDef* ADCx, uint16_t HighThreshold, uint16_t LowThreshold);
void ADC_AnalogWatchdogRegularChannelConfig(ADC_TypeDef* ADCx, uint8_t ADC_Channel);
void ADC_AnalogWatchdogInjectChannelConfig(ADC_TypeDef* ADCx, uint8_t ADC_Channel);

/* Regular Channels Configuration functions */
void ADC_RegularChannelConfig(ADC_TypeDef* ADCx, uint8_t ADC_Channel, uint8_t Rank, uint8_t ADC_SampleTime);
void ADC_SoftwareStartConv(ADC_TypeDef* ADCx);
FlagStatus ADC_GetSoftwareStartConvStatus(ADC_TypeDef* ADCx);
void ADC_ContinuousModeCmd(ADC_TypeDef* ADCx, FunctionalState NewState);
void ADC_DiscModeChannelCountConfig(ADC_TypeDef* ADCx, uint8_t Number);
void ADC_DiscModeCmd(ADC_TypeDef* ADCx, FunctionalState NewState);
uint32_t ADC_GetConversionValue(ADC_TypeDef* ADCx);

/* Temperature Sensor and VBAT management functions */
void ADC_VBATCmd(FunctionalState NewState);                             
void ADC_TempSensorCmd(FunctionalState NewState);

/* Regular Channels DMA Configuration functions */
void ADC_DMACmd(ADC_TypeDef* ADCx, FunctionalState NewState);
            
/* Injected channels Configuration functions */
void ADC_InjectedChannelConfig(ADC_TypeDef* ADCx, uint8_t ADC_Channel, uint8_t Rank, uint8_t ADC_SampleTime);
void ADC_InjectedSequencerLengthConfig(ADC_TypeDef* ADCx, uint8_t Length);
void ADC_ExternalTrigInjectedConvConfig(ADC_TypeDef* ADCx, uint32_t ADC_ExternalTrigInjecConv);
void ADC_ExternalTrigInjectedConvEdgeConfig(ADC_TypeDef* ADCx, uint32_t ADC_ExternalTrigInjecConvEdge);
void ADC_SoftwareStartInjectedConv(ADC_TypeDef* ADCx);
FlagStatus ADC_GetSoftwareStartInjectedConvCmdStatus(ADC_TypeDef* ADCx);
void ADC_AutoInjectedConvCmd(ADC_TypeDef* ADCx, FunctionalState NewState);
void ADC_InjectedDiscModeCmd(ADC_TypeDef* ADCx, FunctionalState NewState);
uint32_t ADC_GetInjectedConversionValue(ADC_TypeDef* ADCx, uint8_t JDRNumber);

/* Interrupts and flags management functions */
void ADC_ITConfig(ADC_TypeDef* ADCx, uint16_t ADC_IT, FunctionalState NewState);  
FlagStatus ADC_GetFlagStatus(ADC_TypeDef* ADCx, uint8_t ADC_FLAG);
void ADC_ClearFlag(ADC_TypeDef* ADCx, uint8_t ADC_FLAG);
ITStatus ADC_GetITStatus(ADC_TypeDef* ADCx, uint16_t ADC_IT);
void ADC_ClearITPendingBit(ADC_TypeDef* ADCx, uint32_t ADC_IT);

                  
/* Multi modal functions */
uint32_t ADC_GetMultiModeConversionValue(void);








  
  

#endif /* __SPL_ADC_H__ */


/************************ (C) COPYRIGHT AISINOCHIP *****END OF FILE****/






