
/******************************************************************************
*@file  : spl_exti.h
*@brief : Header file of EXTI SPL module.
******************************************************************************/

#ifndef __SPL_EXTI_H__
#define __SPL_EXTI_H__

#include "acm32g103_spl_conf.h"

/** 
  * @brief  EXTI Init Structure definition  
  */

typedef struct
{
    GPIO_TypeDef        *GPIOx;     /*!< Specifies the EXTI lines to be enabled or disabled.
                                         This parameter can be any combination value of @ref EXTI_Lines */

    uint32_t            Line;       /*!< Specifies the EXTI lines to be enabled or disabled.
                                         This parameter can be any combination value of @ref EXTI_Lines */

    uint32_t            Mode;       /*!< Specifies the mode for the EXTI lines.
                                         This parameter can be a value of @ref EXTIMode_TypeDef */

    uint32_t            Trigger;    /*!< Specifies the trigger signal active edge for the EXTI lines.
                                         This parameter can be a value of @ref EXTITrigger_TypeDef */

    FunctionalState     Cmd;        /*!< Specifies the new state of the selected EXTI lines.
                                         This parameter can be set either to ENABLE or DISABLE */ 
}EXTI_InitTypeDef;

/** @defgroup EXTI_Line
  * @brief    When line is 0-15, its definition is the same as GPIO_pins
  * @{
  */

#define EXTI_LINE_0				                ( BIT0 )
#define EXTI_LINE_1				                ( BIT1 )
#define EXTI_LINE_2				                ( BIT2 )
#define EXTI_LINE_3				                ( BIT3 )
#define EXTI_LINE_4				                ( BIT4 )
#define EXTI_LINE_5				                ( BIT5 )
#define EXTI_LINE_6				                ( BIT6 )
#define EXTI_LINE_7				                ( BIT7 )
#define EXTI_LINE_8				                ( BIT8 )
#define EXTI_LINE_9				                ( BIT9 )
#define EXTI_LINE_10				            ( BIT10 )
#define EXTI_LINE_11				            ( BIT11 )
#define EXTI_LINE_12				            ( BIT12 )
#define EXTI_LINE_13				            ( BIT13 )
#define EXTI_LINE_14				            ( BIT14 )
#define EXTI_LINE_15				            ( BIT15 )
#define EXTI_LINE_LVD				            ( BIT16 )
#define EXTI_LINE_RTC				            ( BIT17 )
#define EXTI_LINE_LPUART1 			            ( BIT18 )
#define EXTI_LINE_IWDT				            ( BIT19 )
#define EXTI_LINE_COMP1				            ( BIT20 )
#define EXTI_LINE_COMP2				            ( BIT21 )
#define EXTI_LINE_USB 				            ( BIT23 )
#define EXTI_LINE_LPTIM1				        ( BIT24 )
#define EXTI_LINE_COMP3				            ( BIT25 )
#define EXTI_LINE_COMP4				            ( BIT26 )

/**
  * @}
  */

/** @brief  PIN mask for assert test
  */

#define EXTI_ALL_LINE_MASK                      (0x7BFFFFFU)    /* PIN mask for assert test */
#define EXTI_GPIO_LINE_MASK                     (0xFFFFU)       /* PIN mask */
#define EXTI_GPIOF_LINE_MASK                    (0x001FU)       /* PIN mask for assert test */

/** @defgroup EXTI_Mode
  * @brief    its definition is the same as that of GPIO_MODE_IT_RISINGGPIO_MODE_IT_FALLING
  *           GPIO_MODE_IT_RISING_FALLINGGPIO_MODE_EVT_RISINGGPIO_MODE_EVT_RISING
  *           GPIO_MODE_EVT_RISING in GPIO_Mode
  * @{
  */

#define  EXTI_MODE_IT                           ( 0U )   /*!< External Interrupt Mode */
#define  EXTI_MODE_EVENT                        ( 1U )   /*!< External Event Mode */

/**
  * @}
  */

/** @defgroup EXTI_Trigger
  * @brief    its definition is the same as that of GPIO_MODE_IT_RISINGGPIO_MODE_IT_FALLING
  *           GPIO_MODE_IT_RISING_FALLINGGPIO_MODE_EVT_RISINGGPIO_MODE_EVT_RISING
  *           GPIO_MODE_EVT_RISING in GPIO_Mode
  * @{
  */

#define  EXTI_TRIGGER_RISING                    (0x10110000U)   /*!< External Interrupt Mode with Rising edge trigger detection          */
#define  EXTI_TRIGGER_FALLING                   (0x10210000U)   /*!< External Interrupt Mode with Falling edge trigger detection         */
#define  EXTI_TRIGGER_RISING_FALLING            (0x10310000U)   /*!< External Interrupt Mode with Rising/Falling edge trigger detection  */

/**
  * @}
  */

/** @defgroup EXTI_GPIOSel  EXTI GPIOSel
  * @brief
  * @{
  */
  
#define EXTI_GPIOA                              ( 0x0U )
#define EXTI_GPIOB                              ( 0x1U )
#define EXTI_GPIOC                              ( 0x2U )
#define EXTI_GPIOD                              ( 0x3U )
#define EXTI_GPIOE                              ( 0x4U )
#define EXTI_GPIOF                              ( 0x5U ) 

/**
  * @}
  */




/** @defgroup  EXTI Private Macros
  * @{
  */
  
#define IS_EXTI_FUNCTIONAL_STATE(_STATE_)                       (((_STATE_) == ENABLE)  || \
                                                                 ((_STATE_) == DISABLE))
  
#define IS_EXTI_ALL_LINE(__LINE__)                              ((((uint32_t)(__LINE__) &  EXTI_ALL_LINE_MASK) != 0x00u) && \
                                                                 (((uint32_t)(__LINE__) & ~EXTI_ALL_LINE_MASK) == 0x00u))

#define IS_EXTI_SINGLE_LINE(__LINE__)                           (((__LINE__) == EXTI_LINE_0) || \
                                                                 ((__LINE__) == EXTI_LINE_1) || \
                                                                 ((__LINE__) == EXTI_LINE_2) || \
                                                                 ((__LINE__) == EXTI_LINE_3) || \
                                                                 ((__LINE__) == EXTI_LINE_4) || \
                                                                 ((__LINE__) == EXTI_LINE_5) || \
                                                                 ((__LINE__) == EXTI_LINE_6) || \
                                                                 ((__LINE__) == EXTI_LINE_7) || \
                                                                 ((__LINE__) == EXTI_LINE_8) || \
                                                                 ((__LINE__) == EXTI_LINE_9) || \
                                                                 ((__LINE__) == EXTI_LINE_10) || \
                                                                 ((__LINE__) == EXTI_LINE_11) || \
                                                                 ((__LINE__) == EXTI_LINE_12) || \
                                                                 ((__LINE__) == EXTI_LINE_13) || \
                                                                 ((__LINE__) == EXTI_LINE_14) || \
                                                                 ((__LINE__) == EXTI_LINE_15) || \
                                                                 ((__LINE__) == EXTI_LINE_LVD) || \
                                                                 ((__LINE__) == EXTI_LINE_RTC) || \
                                                                 ((__LINE__) == EXTI_LINE_LPUART1) || \
                                                                 ((__LINE__) == EXTI_LINE_IWDT) || \
                                                                 ((__LINE__) == EXTI_LINE_COMP1) || \
                                                                 ((__LINE__) == EXTI_LINE_COMP2) || \
                                                                 ((__LINE__) == EXTI_LINE_USB) || \
                                                                 ((__LINE__) == EXTI_LINE_LPTIM1) || \
                                                                 ((__LINE__) == EXTI_LINE_COMP3) || \
                                                                 ((__LINE__) == EXTI_LINE_COMP4))
                                                                 
#define IS_EXTI_GPIOX(__GPIO__)                                 (((__GPIO__) == GPIOA) || \
                                                                 ((__GPIO__) == GPIOB) || \
                                                                 ((__GPIO__) == GPIOC) || \
                                                                 ((__GPIO__) == GPIOD) || \
                                                                 ((__GPIO__) == GPIOE) || \
                                                                 ((__GPIO__) == GPIOF))

#define IS_EXTI_ALL_GPIO_LINE(__GPIO__, __LINE__)               (((__GPIO__) == GPIOF) ? ((((uint32_t)(__LINE__) &  EXTI_GPIOF_LINE_MASK) != 0x00u) &&  \
                                                                                          (((uint32_t)(__LINE__) & ~EXTI_GPIOF_LINE_MASK) == 0x00u))    \
                                                                                       : ((((uint32_t)(__LINE__) &  EXTI_GPIO_LINE_MASK) != 0x00u) &&   \
                                                                                          (((uint32_t)(__LINE__) & ~EXTI_GPIO_LINE_MASK) == 0x00u)))

#define IS_EXTI_SINGLE_GPIO_LINE(__GPIO__, __LINE__)            (((__GPIO__) == GPIOF) ? (((__LINE__) == EXTI_LINE_0)     ||\
                                                                                          ((__LINE__) == EXTI_LINE_1)     ||\
                                                                                          ((__LINE__) == EXTI_LINE_2)     ||\
                                                                                          ((__LINE__) == EXTI_LINE_3)     ||\
                                                                                          ((__LINE__) == EXTI_LINE_4))      \
                                                                                       : (((__LINE__) == EXTI_LINE_0)     ||\
                                                                                          ((__LINE__) == EXTI_LINE_1)     ||\
                                                                                          ((__LINE__) == EXTI_LINE_2)     ||\
                                                                                          ((__LINE__) == EXTI_LINE_3)     ||\
                                                                                          ((__LINE__) == EXTI_LINE_4)     ||\
                                                                                          ((__LINE__) == EXTI_LINE_5)     ||\
                                                                                          ((__LINE__) == EXTI_LINE_6)     ||\
                                                                                          ((__LINE__) == EXTI_LINE_7)     ||\
                                                                                          ((__LINE__) == EXTI_LINE_8)     ||\
                                                                                          ((__LINE__) == EXTI_LINE_9)     ||\
                                                                                          ((__LINE__) == EXTI_LINE_10)    ||\
                                                                                          ((__LINE__) == EXTI_LINE_11)    ||\
                                                                                          ((__LINE__) == EXTI_LINE_12)    ||\
                                                                                          ((__LINE__) == EXTI_LINE_13)    ||\
                                                                                          ((__LINE__) == EXTI_LINE_14)    ||\
                                                                                          ((__LINE__) == EXTI_LINE_15)))


#define IS_EXTI_MODE(__MODE__)		                            (((__MODE__) == EXTI_MODE_IT)   || \
                                                                 ((__MODE__) == EXTI_MODE_EVENT))


#define IS_EXTI_TRIGGER(__TRIGGER__)                            (((__TRIGGER__) == EXTI_TRIGGER_RISING)            || \
                                                                 ((__TRIGGER__) == EXTI_TRIGGER_FALLING)           || \
                                                                 ((__TRIGGER__) == EXTI_TRIGGER_RISING_FALLING))

/**
  * @}
  */

/* Exported functions --------------------------------------------------------*/


void EXTI_Init(EXTI_InitTypeDef* EXTI_InitStruct);
void EXTI_DeInit(void);
void EXTI_StructInit(EXTI_InitTypeDef* EXTI_InitStruct);
void HAL_EXTI_GenerateSWI(uint32_t Line);
FlagStatus EXTI_GetFlagStatus(uint32_t Line);
void EXTI_ClearFlag(uint32_t Line);
void EXTI_ClearAllFlag(void);
FlagStatus EXTI_GetITStatus(uint32_t Line);
void EXTI_ClearITPendingBit(uint32_t Line);
void EXTI_ClearAllITPendingBit(void);





#endif

