
/******************************************************************************
*@file  : spl_i2s.h
*@brief : Header file of I2S SPL module.
******************************************************************************/

#ifndef __SPL_I2S_H__
#define __SPL_I2S_H__

#include "acm32g103_spl_conf.h"

/**
  * @brief I2S Init structure definition
  */
  
typedef struct
{
    uint32_t Mode;                          /*!< Specifies the I2S operating mode.
                                                  This parameter can be a value of @ref I2S_Mode            */
    
    uint32_t Standard;                      /*!< Specifies the standard used for the I2S communication.
                                                  This parameter can be a value of @ref I2S_Standard        */
    
    uint32_t DataFormat;                    /*!< Specifies the data format for the I2S communication.
                                                  This parameter can be a value of @ref I2S_DataFormat      */
    
    FunctionalState MCLKOutput;             /*!< Specifies whether the I2S MCLK output is enabled or not.
                                                  This parameter can be: ENABLE or DISABLE.                 */

    uint32_t ClockPolarity;                 /*!< Specifies the idle state of the I2S clock.
                                                  This parameter can be a value of @ref I2S_ClockPolarity   */

    uint32_t AudioFreq;                    /*!< Specifies the frequency selected for the I2S communication. */
    
}I2S_InitTypeDef;


/** @defgroup I2S_Mode
  * @{
  */
  
#define I2S_MODE_SLAVE_TX                   ( 0U )
#define I2S_MODE_SLAVE_RX                   ( I2S_CR_MODE_0 )
#define I2S_MODE_MASTER_TX                  ( I2S_CR_MODE_1 )
#define I2S_MODE_MASTER_RX                  ( I2S_CR_MODE_1 | I2S_CR_MODE_0)

/**
  * @}
  */


/** @defgroup I2S_Standard
  * @{
  */
  
#define I2S_STANDARD_PHILIPS                ( 0U )
#define I2S_STANDARD_MSB                    ( I2S_CR_STD_0 )
#define I2S_STANDARD_LSB                    ( I2S_CR_STD_1 )
#define I2S_STANDARD_PCM_SHORT              ( I2S_CR_STD_1 | I2S_CR_STD_0 )
#define I2S_STANDARD_PCM_LONG               ( I2S_CR_PCMSMOD | I2S_CR_STD_1 | I2S_CR_STD_0 )

/**
  * @}
  */


/** @defgroup I2S_DataFormat
  * @{
  */
  
#define I2S_DATAFORMAT_16B_EXTENDED_TO_16B  ( 0U )
#define I2S_DATAFORMAT_16B_EXTENDED_TO_32B  ( I2S_CR_CHLEN )
#define I2S_DATAFORMAT_24B_EXTENDED_TO_32B  ( I2S_CR_CHLEN | I2S_CR_DTLEN_0 )
#define I2S_DATAFORMAT_32B_EXTENDED_TO_32B  ( I2S_CR_CHLEN | I2S_CR_DTLEN_1 )

/**
  * @}
  */

/** @defgroup I2S_ClockPolarity
  * @{
  */
  
#define I2S_CLOCKPOLARITY_LOW               ( 0U )
#define I2S_CLOCKPOLARITY_HIGH              ( I2S_CR_CKPL )

/**
  * @}
  */

/** @defgroup I2S_IT
  * @{
  */
  
#define I2S_IT_TXE                          ( I2S_DIER_TBEIE )
#define I2S_IT_RXNE                         ( I2S_DIER_RBNEIE )
#define I2S_IT_ERR                          ( I2S_DIER_ERRIE )

/**
  * @}
  */

/** 
  * @brief  
  */

#define I2S_IT_MASK                         ( I2S_IT_TXE | I2S_IT_RXNE | I2S_IT_ERR )

/** @defgroup I2S_Flags
  * @{
  */
  
#define I2S_FLAG_FE                         ( I2S_SR_FE )
#define I2S_FLAG_BUSY                       ( I2S_SR_BUSY )
#define I2S_FLAG_RXOVE                      ( I2S_SR_RXOVE )
#define I2S_FLAG_TXUDE                      ( I2S_SR_TXUDE )
#define I2S_FLAG_CHF                        ( I2S_SR_CHF )
#define I2S_FLAG_TXE                        ( I2S_SR_TXE )
#define I2S_FLAG_RXNE                       ( I2S_SR_RXNE )

/**
  * @}
  */

/** 
  * @brief  
  */

#define I2S_FLAG_MASK                       ( I2S_FLAG_FE | I2S_FLAG_BUSY | I2S_FLAG_RXOVE | I2S_FLAG_TXUDE | \
                                              I2S_FLAG_CHF | I2S_FLAG_TXE | I2S_FLAG_RXNE )

#define I2S_CLEAR_FLAG_MASK                 ( I2S_FLAG_FE )
    

/** @defgroup  I2S Private Macros
  * @{
  */
  
#define IS_I2S_ALL_PERIPH(__I2S__)                                      (((__I2S__) == I2S1) || \
                                                                         ((__I2S__) == I2S2))

#define IS_I2S_FUNCTIONAL_STATE(__STATE__)                              (((__STATE__) == ENABLE)    || \
                                                                         ((__STATE__) == DISABLE))

#define IS_I2S_MODE(__MODE__)                                           (((__MODE__) == I2S_MODE_SLAVE_TX)  || \
                                                                         ((__MODE__) == I2S_MODE_SLAVE_RX)  || \
                                                                         ((__MODE__) == I2S_MODE_MASTER_TX) || \
                                                                         ((__MODE__) == I2S_MODE_MASTER_RX))

#define IS_I2S_STANDARD(__STANDARD__)                                   (((__STANDARD__) == I2S_STANDARD_PHILIPS)   || \
                                                                         ((__STANDARD__) == I2S_STANDARD_MSB)       || \
                                                                         ((__STANDARD__) == I2S_STANDARD_LSB)       || \
                                                                         ((__STANDARD__) == I2S_STANDARD_PCM_SHORT) || \
                                                                         ((__STANDARD__) == I2S_STANDARD_PCM_LONG))

#define IS_I2S_DATAFORMAT(__DATAFORMAT__)                               (((__DATAFORMAT__) == I2S_DATAFORMAT_16B_EXTENDED_TO_16B) || \
                                                                         ((__DATAFORMAT__) == I2S_DATAFORMAT_16B_EXTENDED_TO_32B) || \
                                                                         ((__DATAFORMAT__) == I2S_DATAFORMAT_24B_EXTENDED_TO_32B) || \
                                                                         ((__DATAFORMAT__) == I2S_DATAFORMAT_32B_EXTENDED_TO_32B))

#define IS_I2S_CLOCKPOLARITY(__POLARITY__)                              (((__POLARITY__) == I2S_CLOCKPOLARITY_LOW)  || \
                                                                         ((__POLARITY__) == I2S_CLOCKPOLARITY_HIGH))

#define IS_I2S_ALL_IT(__IT__)                                           ((((uint32_t)(__IT__) &  I2S_IT_MASK) != 0x00u) && \
                                                                         (((uint32_t)(__IT__) & ~I2S_IT_MASK) == 0x00u))
                                                                 
#define IS_I2S_SINGLE_IT(__IT__)                                        (((__IT__) == I2S_IT_TXE) || \
                                                                         ((__IT__) == I2S_IT_RXNE) || \
                                                                         ((__IT__) == I2S_IT_ERR))
                                                                 
#define IS_I2S_ALL_FLAG(__FLAG__)                                       ((((uint32_t)(__FLAG__) &  I2S_FLAG_MASK) != 0x00u) && \
                                                                         (((uint32_t)(__FLAG__) & ~I2S_FLAG_MASK) == 0x00u))
                                                                 
#define IS_I2S_SINGLE_FLAG(__FLAG__)                                    (((__FLAG__) == I2S_FLAG_FE) || \
                                                                         ((__FLAG__) == I2S_FLAG_BUSY) || \
                                                                         ((__FLAG__) == I2S_FLAG_RXOVE) || \
                                                                         ((__FLAG__) == I2S_FLAG_TXUDE) || \
                                                                         ((__FLAG__) == I2S_FLAG_CHF) || \
                                                                         ((__FLAG__) == I2S_FLAG_TXE) || \
                                                                         ((__FLAG__) == I2S_FLAG_RXNE))
                                                                 
#define IS_I2S_CLEAR_FLAG(__FLAG__)                                    ((((uint32_t)(__FLAG__) &  I2S_CLEAR_FLAG_MASK) != 0x00u) && \
                                                                         (((uint32_t)(__FLAG__) & ~I2S_CLEAR_FLAG_MASK) == 0x00u))
                                                                 
                                        





/**
  * @}
  */

/* Exported functions --------------------------------------------------------*/


void I2S_Init(I2S_TypeDef *I2Sx, I2S_InitTypeDef *I2S_InitStruct);
void I2S_DeInit(I2S_TypeDef *I2Sx);
void I2S_Reset(I2S_TypeDef *I2Sx);
void I2S_Cmd(I2S_TypeDef* I2Sx, FunctionalState NewState);
void I2S_TxDMACmd(I2S_TypeDef* I2Sx, FunctionalState NewState);
void I2S_RxDMACmd(I2S_TypeDef* I2Sx, FunctionalState NewState);
void I2S_Reload(I2S_TypeDef *I2Sx);
void I2S_ConfigMode(I2S_TypeDef* I2Sx, uint32_t Mode);
void I2S_ConfigStandard(I2S_TypeDef* I2Sx, uint32_t Standard);
void I2S_ConfigDataFormat(I2S_TypeDef* I2Sx, uint32_t DataFormat);
void I2S_ConfigClockPolarity(I2S_TypeDef* I2Sx, uint32_t ClockPolarity);
void I2S_MCLKOutputCmd(I2S_TypeDef* I2Sx, FunctionalState NewState);
void I2S_ConfigAudioFreq(I2S_TypeDef* I2Sx, uint32_t AudioFreq, FunctionalState MCLKOutput, uint32_t DataFormat);
void I2S_ITConfig(I2S_TypeDef* I2Sx, uint32_t IT, FunctionalState NewState);
FlagStatus I2S_GetFlagStatus(I2S_TypeDef* I2Sx, uint32_t Flag);
void I2S_ClearFlag(I2S_TypeDef* I2Sx, uint32_t Flag);





#endif

