
/******************************************************************************
*@file  : spl_lptim.h
*@brief : Header file of LPTIM SPL module.
******************************************************************************/

#ifndef __SPL_LPTIM_H__
#define __SPL_LPTIM_H__ 

#include "acm32g103_spl_conf.h"

/**
  * @brief  LPTIM Initialization Structure definition
  */

typedef struct
{
    uint32_t        ClockSource;        /*!< Selects the clock source.
                                             This parameter can be a value of @ref LPTIM_ClockSource         */

    uint32_t        Prescaler;          /*!< Specifies the timer clock Prescaler.
                                             This parameter can be a value of @ref LPTIM_ClockPrescaler      */

    uint32_t        Waveform;           /*!< Selects the output shape.
                                             This parameter can be a value of @ref LPTIM_Waveform */

    uint32_t        OutputPolarity;     /*!< Specifies the Output polarity.
                                             This parameter can be a value of @ref LPTIM_OutputPolarity */

} LPTIM_InitTypeDef;


/** @defgroup LPTIM_ClockSource
  * @{
  */
  
#define LPTIM_CLOCKSOURCE_INTERNAL_PCLK                         ( RCC_LPTIM1CLKSOURCE_PCLK )
#define LPTIM_CLOCKSOURCE_INTERNAL_RC32K                        ( RCC_LPTIM1CLKSOURCE_RC32K )
#define LPTIM_CLOCKSOURCE_INTERNAL_RC64M                        ( RCC_LPTIM1CLKSOURCE_RC64M )
#define LPTIM_CLOCKSOURCE_INTERNAL_XTL                          ( RCC_LPTIM1CLKSOURCE_XTL )
#define LPTIM_CLOCKSOURCE_EXTERNAL_INPUT1                       ( LPTIM_CFGR1_CKSEL )

/**
  * @}
  */

/** @defgroup LPTIM_ClockPrescaler
  * @{
  */
  
#define LPTIM_CLOCKPRESCALER_DIV1                               ( 0U )
#define LPTIM_CLOCKPRESCALER_DIV2                               ( LPTIM_CFGR1_PRESC_0 )
#define LPTIM_CLOCKPRESCALER_DIV4                               ( LPTIM_CFGR1_PRESC_1 )
#define LPTIM_CLOCKPRESCALER_DIV8                               ( LPTIM_CFGR1_PRESC_0 | LPTIM_CFGR1_PRESC_1 )
#define LPTIM_CLOCKPRESCALER_DIV16                              ( LPTIM_CFGR1_PRESC_2 )
#define LPTIM_CLOCKPRESCALER_DIV32                              ( LPTIM_CFGR1_PRESC_0 | LPTIM_CFGR1_PRESC_2 )
#define LPTIM_CLOCKPRESCALER_DIV64                              ( LPTIM_CFGR1_PRESC_1 | LPTIM_CFGR1_PRESC_2 )
#define LPTIM_CLOCKPRESCALER_DIV128                             ( LPTIM_CFGR1_PRESC )

/**
  * @}
  */

/** @defgroup LPTIM_Waveform
  * @{
  */
  
#define LPTIM_WAVEFORM_PWM_ONEPULSE                             ( 0U )
#define LPTIM_WAVEFORM_SETONCE                                  ( LPTIM_CFGR1_WAVE )

/**
  * @}
  */

/** @defgroup LPTIM_OutputPolarity
  * @{
  */

#define LPTIM_OUTPUTPOLARITY_HIGH                               ( 0U )
#define LPTIM_OUTPUTPOLARITY_LOW                                ( LPTIM_CFGR1_WAVPOL )

/**
  * @}
  */
/** @defgroup LPTIM_ClockPolarity
  * @{
  */
  
#define LPTIM_CLOCKPOLARITY_RISING                              ( 0U )
#define LPTIM_CLOCKPOLARITY_FALLING                             ( LPTIM_CFGR1_CKPOL_0 )
#define LPTIM_CLOCKPOLARITY_RISING_FALLING                      ( LPTIM_CFGR1_CKPOL_1 )

/**
  * @}
  */

/** @defgroup LPTIM_CounterFilter
  * @{
  */
  
#define LPTIM_COUNTERFILTER_1CLK                                ( 0U )
#define LPTIM_COUNTERFILTER_2CLK                                ( LPTIM_CFGR1_CKFLT_0 )
#define LPTIM_COUNTERFILTER_4CLK                                ( LPTIM_CFGR1_CKFLT_1 )
#define LPTIM_COUNTERFILTER_8CLK                                ( LPTIM_CFGR1_CKFLT )

/**
  * @}
  */

/** @defgroup LPTIM_TriggerSource
  * @{
  */
  
#define LPTIM_TRIGSOURCE_SOFTWARE                               ( 0x0000FFFFU )
#define LPTIM_TRIGSOURCE_0                                      ( 0U )
#define LPTIM_TRIGSOURCE_1                                      ( LPTIM_CFGR1_TRIGSEL_0 )
#define LPTIM_TRIGSOURCE_2                                      ( LPTIM_CFGR1_TRIGSEL_1 )
#define LPTIM_TRIGSOURCE_3                                      ( LPTIM_CFGR1_TRIGSEL_1 | LPTIM_CFGR1_TRIGSEL_0 )
#define LPTIM_TRIGSOURCE_4                                      ( LPTIM_CFGR1_TRIGSEL_2 )
#define LPTIM_TRIGSOURCE_5                                      ( LPTIM_CFGR1_TRIGSEL_2 | LPTIM_CFGR1_TRIGSEL_0 )
#define LPTIM_TRIGSOURCE_6                                      ( LPTIM_CFGR1_TRIGSEL_2 | LPTIM_CFGR1_TRIGSEL_1 )
#define LPTIM_TRIGSOURCE_7                                      ( LPTIM_CFGR1_TRIGSEL_2 | LPTIM_CFGR1_TRIGSEL_1 | LPTIM_CFGR1_TRIGSEL_0 )

/**
  * @}
  */

/** @defgroup LPTIM_TriggerPolarity
  * @{
  */
  
#define LPTIM_TRIGPOLARITY_RISING                               ( LPTIM_CFGR1_TRIGEN_0 )
#define LPTIM_TRIGPOLARITY_FALLING                              ( LPTIM_CFGR1_TRIGEN_1 )
#define LPTIM_TRIGPOLARITY_RISING_FALLING                       ( LPTIM_CFGR1_TRIGEN_1 | LPTIM_CFGR1_TRIGEN_0 )

/**
  * @}
  */

/** @defgroup LPTIM_TriggerFilter
  * @{
  */
  
#define LPTIM_TRIGFILTER_1CLK                                   ( 0U )
#define LPTIM_TRIGFILTER_2CLK                                   ( LPTIM_CFGR1_TRIGFLT_0 )
#define LPTIM_TRIGFILTER_4CLK                                   ( LPTIM_CFGR1_TRIGFLT_1 )
#define LPTIM_TRIGFILTER_8CLK                                   ( LPTIM_CFGR1_TRIGFLT_1 | LPTIM_CFGR1_TRIGFLT_0 )

/**
  * @}
  */

/** @defgroup LPTIM_Mode
  * @{
  */
  
#define LPTIM_MODE_CONTINUOUS                                   ( LPTIM_CR_CNTSTRT )
#define LPTIM_MODE_SINGLE                                       ( LPTIM_CR_SNGSTRT )

/**
  * @}
  */




/** @defgroup LPTIM_UpdatingMode
  * @{
  */

#define LPTIM_UPDATE_IMMEDIATE                                  ( 0U )
#define LPTIM_UPDATE_ENDOFPERIOD                                ( LPTIM_CFGR1_PRELOAD )

/**
  * @}
  */

/** @defgroup LPTIM_CounterSource
  * @{
  */

#define LPTIM_COUNTERSOURCE_INTERNAL                            ( 0U )
#define LPTIM_COUNTERSOURCE_EXTERNAL                            ( LPTIM_CFGR1_COUNTMODE )

/**
  * @}
  */

/** @defgroup LPTIM_Input1_Source LPTIM Input1 Source
  * @{
  */

#define LPTIM_INPUT1SOURCE_0                                    ( 0U )
#define LPTIM_INPUT1SOURCE_1                                    ( LPTIM_CFGR2_IN1SEL_0 )
#define LPTIM_INPUT1SOURCE_2                                    ( LPTIM_CFGR2_IN1SEL_1 )
#define LPTIM_INPUT1SOURCE_3                                    ( LPTIM_CFGR2_IN1SEL_1 | LPTIM_CFGR2_IN1SEL_0 )

/**
  * @}
  */

/** @defgroup LPTIM_Input2_Source LPTIM Input2 Source
  * @{
  */

#define LPTIM_INPUT2SOURCE_0                                    ( 0U )
#define LPTIM_INPUT2SOURCE_1                                    ( LPTIM_CFGR2_IN2SEL_0 )
#define LPTIM_INPUT2SOURCE_2                                    ( LPTIM_CFGR2_IN2SEL_1 )
#define LPTIM_INPUT2SOURCE_3                                    ( LPTIM_CFGR2_IN2SEL_1 | LPTIM_CFGR2_IN2SEL_0 )

/**
  * @}
  */


/** @defgroup LPTIM_Flags
  * @{
  */

#define LPTIM_FLAG_REPOK                                        ( LPTIM_ISR_REPOK )
#define LPTIM_FLAG_REPUE                                        ( LPTIM_ISR_REPUE )
#define LPTIM_FLAG_DOWN                                         ( LPTIM_ISR_DOWN )
#define LPTIM_FLAG_UP                                           ( LPTIM_ISR_UP )
#define LPTIM_FLAG_ARROK                                        ( LPTIM_ISR_ARROK )
#define LPTIM_FLAG_CMPOK                                        ( LPTIM_ISR_CMPOK )
#define LPTIM_FLAG_EXTTRIG                                      ( LPTIM_ISR_EXTTRIG )
#define LPTIM_FLAG_ARRM                                         ( LPTIM_ISR_ARRM )
#define LPTIM_FLAG_CMPM                                         ( LPTIM_ISR_CMPM )

/**
  * @}
  */

/** @brief  IT mask for assert test
  */
#define LPTIM_FLAG_MASK                                           ( LPTIM_FLAG_REPOK | LPTIM_FLAG_REPUE | LPTIM_FLAG_DOWN | \
                                                                  LPTIM_FLAG_UP | LPTIM_FLAG_ARROK | LPTIM_FLAG_CMPOK   | \
                                                                  LPTIM_FLAG_EXTTRIG | LPTIM_FLAG_ARRM | LPTIM_FLAG_CMPM )
          
/** @defgroup LPTIM_IT
  * @{
  */
  
#define LPTIM_IT_REPOK                                          ( LPTIM_IER_REPOKIE )
#define LPTIM_IT_REPUE                                          ( LPTIM_IER_UEIE )
#define LPTIM_IT_DOWN                                           ( LPTIM_IER_DOWNIE )
#define LPTIM_IT_UP                                             ( LPTIM_IER_UPIE )
#define LPTIM_IT_ARROK                                          ( LPTIM_IER_ARROKIE )
#define LPTIM_IT_CMPOK                                          ( LPTIM_IER_CMPOKIE )
#define LPTIM_IT_EXTTRIG                                        ( LPTIM_IER_EXTTRIGIE )
#define LPTIM_IT_ARRM                                           ( LPTIM_IER_ARRMIE )
#define LPTIM_IT_CMPM                                           ( LPTIM_IER_CMPMIE )

/**
  * @}
  */

/** @brief  IT mask for assert test
  */
#define LPTIM_IT_MASK                                           ( LPTIM_IT_REPOK | LPTIM_IT_REPUE | LPTIM_IT_DOWN | \
                                                                  LPTIM_IT_UP | LPTIM_IT_ARROK | LPTIM_IT_CMPOK   | \
                                                                  LPTIM_IT_EXTTRIG | LPTIM_IT_ARRM | LPTIM_IT_CMPM )
                                                                  
                                                                  


/* Private macros ------------------------------------------------------------*/
/** @defgroup LPTIM_Private_Macros LPTIM Private Macros
  * @{
  */

#define IS_LPTIM_ALL_PERIPH(__PERIPH__)                         ((__PERIPH__) == LPTIM1)

#define IS_LPTIM_FUNCTIONAL_STATE(_STATE_)                      (((_STATE_) == ENABLE)      || \
                                                                 ((_STATE_) == DISABLE))

#define IS_LPTIM_CLOCK_SOURCE(__SOURCE__)                       (((__SOURCE__) == LPTIM_CLOCKSOURCE_INTERNAL_PCLK) || \
                                                                 ((__SOURCE__) == LPTIM_CLOCKSOURCE_INTERNAL_RC32K)  || \
                                                                 ((__SOURCE__) == LPTIM_CLOCKSOURCE_INTERNAL_RC64M)  || \
                                                                 ((__SOURCE__) == LPTIM_CLOCKSOURCE_INTERNAL_XTL)  || \
                                                                 ((__SOURCE__) == LPTIM_CLOCKSOURCE_EXTERNAL_INPUT1))

#define IS_LPTIM_CLOCK_POLARITY(__POLARITY__)                   (((__POLARITY__) == LPTIM_CLOCKPOLARITY_RISING) || \
                                                                 ((__POLARITY__) == LPTIM_CLOCKPOLARITY_FALLING)  || \
                                                                 ((__POLARITY__) == LPTIM_CLOCKPOLARITY_RISING_FALLING))
                                                                 
#define IS_LPTIM_CLOCK_PRESCALER(__PRESCALER__)                 (((__PRESCALER__) ==  LPTIM_CLOCKPRESCALER_DIV1)  || \
                                                                 ((__PRESCALER__) ==  LPTIM_CLOCKPRESCALER_DIV2)  || \
                                                                 ((__PRESCALER__) ==  LPTIM_CLOCKPRESCALER_DIV4)  || \
                                                                 ((__PRESCALER__) ==  LPTIM_CLOCKPRESCALER_DIV8)  || \
                                                                 ((__PRESCALER__) ==  LPTIM_CLOCKPRESCALER_DIV16) || \
                                                                 ((__PRESCALER__) ==  LPTIM_CLOCKPRESCALER_DIV32) || \
                                                                 ((__PRESCALER__) ==  LPTIM_CLOCKPRESCALER_DIV64) || \
                                                                 ((__PRESCALER__) ==  LPTIM_CLOCKPRESCALER_DIV128))

#define IS_LPTIM_TRIG_SOURCE(__TRIG__)                          (((__TRIG__) == LPTIM_TRIGSOURCE_SOFTWARE) || \
                                                                 ((__TRIG__) == LPTIM_TRIGSOURCE_0)        || \
                                                                 ((__TRIG__) == LPTIM_TRIGSOURCE_1)        || \
                                                                 ((__TRIG__) == LPTIM_TRIGSOURCE_2)        || \
                                                                 ((__TRIG__) == LPTIM_TRIGSOURCE_3)        || \
                                                                 ((__TRIG__) == LPTIM_TRIGSOURCE_4)        || \
                                                                 ((__TRIG__) == LPTIM_TRIGSOURCE_5)        || \
                                                                 ((__TRIG__) == LPTIM_TRIGSOURCE_6)        || \
                                                                 ((__TRIG__) == LPTIM_TRIGSOURCE_7))

  
#define IS_LPTIM_TRIG_POLARITY(__POLARITY__)                    (((__POLARITY__) == LPTIM_TRIGPOLARITY_RISING)  || \
                                                                 ((__POLARITY__) == LPTIM_TRIGPOLARITY_FALLING) || \
                                                                 ((__POLARITY__) == LPTIM_TRIGPOLARITY_RISING_FALLING))
  
  
#define IS_LPTIM_TRIG_FILTER(__FILTER__)                        (((__FILTER__) == LPTIM_TRIGFILTER_1CLK) || \
                                                                 ((__FILTER__) == LPTIM_TRIGFILTER_2CLK)    || \
                                                                 ((__FILTER__) == LPTIM_TRIGFILTER_4CLK)    || \
                                                                 ((__FILTER__) == LPTIM_TRIGFILTER_8CLK))
  
#define IS_LPTIM_COUNTER_SOURCE(__SOURCE__)                     (((__SOURCE__) == LPTIM_COUNTERSOURCE_INTERNAL) || \
                                                                 ((__SOURCE__) == LPTIM_COUNTERSOURCE_EXTERNAL))

  
     

#define IS_LPTIM_COUNTER_FILTER(__FILTER__)                     (((__FILTER__) == LPTIM_COUNTERFILTER_1CLK)    || \
                                                                 ((__FILTER__) == LPTIM_COUNTERFILTER_2CLK)    || \
                                                                 ((__FILTER__) == LPTIM_COUNTERFILTER_4CLK)    || \
                                                                 ((__FILTER__) == LPTIM_COUNTERFILTER_8CLK))

#define IS_LPTIM_MODE(__MODE__)                                 (((__MODE__) == LPTIM_MODE_CONTINUOUS)    || \
                                                                 ((__MODE__) == LPTIM_MODE_SINGLE))
  

#define IS_LPTIM_WAVEFORM(__WAVEFORM__)                         (((__WAVEFORM__) == LPTIM_WAVEFORM_PWM_ONEPULSE)    || \
                                                                 ((__WAVEFORM__) == LPTIM_WAVEFORM_SETONCE))
  
#define IS_LPTIM_OUTPUT_POLARITY(__POLARITY__)                  (((__POLARITY__) == LPTIM_OUTPUTPOLARITY_HIGH) || \
                                                                 ((__POLARITY__) == LPTIM_OUTPUTPOLARITY_LOW))


#define IS_LPTIM_UPDATE_MODE(__MODE__)                          (((__MODE__) == LPTIM_UPDATE_IMMEDIATE) || \
                                                                 ((__MODE__) == LPTIM_UPDATE_ENDOFPERIOD))

#define IS_LPTIM_AUTORELOAD(__AUTORELOAD__)                     (((__AUTORELOAD__) <= 0x0000FFFFUL) && ((__AUTORELOAD__) != 0UL))

#define IS_LPTIM_COMPARE(__COMPARE__)                           ((__COMPARE__) < 0x0000FFFFUL)

#define IS_LPTIM_PERIOD(__PERIOD__)                             IS_LPTIM_AUTORELOAD(__PERIOD__)

#define IS_LPTIM_PULSE(__PULSE__)                               IS_LPTIM_COMPARE(__PULSE__)

#define IS_LPTIM_TIMEOUT(__TIMEOUT__)                           IS_LPTIM_COMPARE(__TIMEOUT__)

#define IS_LPTIM_REPETITION_COUNTER(__COUNTER__)                ((__COUNTER__) <= 0xFFUL)

#define IS_LPTIM_INPUT1_SOURCE(__SOURCE__)                      (((__SOURCE__) == LPTIM_INPUT1SOURCE_0) || \
                                                                 ((__SOURCE__) == LPTIM_INPUT1SOURCE_1) || \
                                                                 ((__SOURCE__) == LPTIM_INPUT1SOURCE_2) || \
                                                                 ((__SOURCE__) == LPTIM_INPUT1SOURCE_3))


#define IS_LPTIM_INPUT2_SOURCE(__SOURCE__)                      (((__SOURCE__) == LPTIM_INPUT2SOURCE_0) || \
                                                                 ((__SOURCE__) == LPTIM_INPUT2SOURCE_1) || \
                                                                 ((__SOURCE__) == LPTIM_INPUT2SOURCE_2) || \
                                                                 ((__SOURCE__) == LPTIM_INPUT2SOURCE_3))
                                                                 
#define IS_LPTIM_ALL_IT(__IT__)                                 ((((uint32_t)(__IT__) &  LPTIM_IT_MASK) != 0x00u) && \
                                                                 (((uint32_t)(__IT__) & ~LPTIM_IT_MASK) == 0x00u))
                                                                 
#define IS_LPTIM_SINGLE_IT(__IT__)                              (((__IT__) == LPTIM_IT_REPOK) || \
                                                                 ((__IT__) == LPTIM_IT_REPUE) || \
                                                                 ((__IT__) == LPTIM_IT_DOWN) || \
                                                                 ((__IT__) == LPTIM_IT_UP) || \
                                                                 ((__IT__) == LPTIM_IT_ARROK) || \
                                                                 ((__IT__) == LPTIM_IT_CMPOK) || \
                                                                 ((__IT__) == LPTIM_IT_EXTTRIG) || \
                                                                 ((__IT__) == LPTIM_IT_ARRM) || \
                                                                 ((__IT__) == LPTIM_IT_CMPM))
                                                                 
#define IS_LPTIM_ALL_FLAG(__FLAG__)                             ((((uint32_t)(__FLAG__) &  LPTIM_FLAG_MASK) != 0x00u) && \
                                                                 (((uint32_t)(__FLAG__) & ~LPTIM_FLAG_MASK) == 0x00u))
                                                                 
#define IS_LPTIM_SINGLE_FLAG(__FLAG__)                          (((__FLAG__) == LPTIM_FLAG_REPOK) || \
                                                                 ((__FLAG__) == LPTIM_FLAG_REPUE) || \
                                                                 ((__FLAG__) == LPTIM_FLAG_DOWN) || \
                                                                 ((__FLAG__) == LPTIM_FLAG_UP) || \
                                                                 ((__FLAG__) == LPTIM_FLAG_ARROK) || \
                                                                 ((__FLAG__) == LPTIM_FLAG_CMPOK) || \
                                                                 ((__FLAG__) == LPTIM_FLAG_EXTTRIG) || \
                                                                 ((__FLAG__) == LPTIM_FLAG_ARRM) || \
                                                                 ((__FLAG__) == LPTIM_FLAG_CMPM))
                                                                 
                                                                 


/**
  * @}
  */

/* Exported functions --------------------------------------------------------*/

void LPTIM_DeInit(LPTIM_TypeDef *LPTIMx);
void LPTIM_Reset(LPTIM_TypeDef *LPTIMx);
void LPTIM_Init(LPTIM_TypeDef *LPTIMx, LPTIM_InitTypeDef *LPTIM_InitStruct);
void LPTIM_StructInit(LPTIM_InitTypeDef *LPTIM_InitStruct);
void LPTIM_Cmd(LPTIM_TypeDef* LPTIMx, FunctionalState NewState);
void LPTIM_SelectClockSource(LPTIM_TypeDef* LPTIMx, uint32_t ClockSource);
void LPTIM_ConfigPrescaler(LPTIM_TypeDef* LPTIMx, uint32_t Prescaler);
void LPTIM_ConfigClockGlitchFilter(LPTIM_TypeDef* LPTIMx, uint32_t Filter);
void LPTIM_ConfigOutputPolarity(LPTIM_TypeDef* LPTIMx, uint32_t OutputPolarity);
void LPTIM_SelectULPTIMClockPolarity(LPTIM_TypeDef* LPTIMx, uint32_t Polarity);
void LPTIM_ConfigExternalTrigger(LPTIM_TypeDef* LPTIMx, uint32_t Source, uint32_t Polarity);
void LPTIM_SelectSoftwareStart(LPTIM_TypeDef* LPTIMx);
void LPTIM_ConfigTriggerGlitchFilter(LPTIM_TypeDef* LPTIMx, uint32_t Filter);
void LPTIM_SelectOperatingMode(LPTIM_TypeDef* LPTIMx, uint32_t Mode);
void LPTIM_TimoutCmd(LPTIM_TypeDef* LPTIMx, FunctionalState NewState);
void LPTIM_ConfigWaveform(LPTIM_TypeDef* LPTIMx, uint32_t Waveform);
void LPTIM_ConfigUpdate(LPTIM_TypeDef* LPTIMx, uint32_t UpdateMode);
void LPTIM_SetAutoreloadValue(LPTIM_TypeDef* LPTIMx, uint32_t Autoreload);
void LPTIM_SetCompareValue(LPTIM_TypeDef* LPTIMx, uint32_t Compare);
void LPTIM_SetRepetitionCounterValue(LPTIM_TypeDef* LPTIMx, uint32_t RepetitionCounter);
void LPTIM_SelectCounterMode(LPTIM_TypeDef* LPTIMx, FunctionalState NewState);
void LPTIM_SelectEncoderMode(LPTIM_TypeDef* LPTIMx, FunctionalState NewState);
uint32_t LPTIM_GetCounterValue(LPTIM_TypeDef* LPTIMx);
uint32_t LPTIM_GetAutoreloadValue(LPTIM_TypeDef* LPTIMx);
uint32_t LPTIM_GetCompareValue(LPTIM_TypeDef* LPTIMx);
uint32_t LPTIM_GetRepetitionCounterValue(LPTIM_TypeDef* LPTIMx);
void LPTIM_Input1RemapConfig(LPTIM_TypeDef* LPTIMx, uint32_t Input1);
void LPTIM_Input2RemapConfig(LPTIM_TypeDef* LPTIMx, uint32_t Input2);
void LPTIM_ResetCounterAfterReadCounter(LPTIM_TypeDef* LPTIMx, FunctionalState NewState);
void LPTIM_ResetCounter(LPTIM_TypeDef* LPTIMx);
FlagStatus LPTIM_GetResetCounterState(LPTIM_TypeDef* LPTIMx);
void LPTIM_ITConfig(LPTIM_TypeDef* LPTIMx, uint32_t IT, FunctionalState NewState);
FlagStatus LPTIM_GetFlagStatus(LPTIM_TypeDef* LPTIMx, uint32_t Flag);
void LPTIM_ClearFlag(LPTIM_TypeDef* LPTIMx, uint32_t Flag);
FlagStatus LPTIM_GetITStatus(LPTIM_TypeDef* LPTIMx, uint32_t IT);

#endif
