
/******************************************************************************
*@file  : spl_rcc.h
*@brief : Header file of RCC SPL module.
******************************************************************************/

#ifndef __SPL_RCC_H__
#define __SPL_RCC_H__

#include "acm32g103_spl_conf.h"
    
typedef struct
{
    uint32_t SYSCLK_Frequency;                  /*!<  SYSCLK clock frequency expressed in Hz */
    uint32_t HCLK_Frequency;                    /*!<  HCLK clock frequency expressed in Hz   */
    uint32_t PCLK1_Frequency;                   /*!<  PCLK1 clock frequency expressed in Hz  */
    uint32_t PCLK2_Frequency;                   /*!<  PCLK2 clock frequency expressed in Hz  */
}RCC_ClocksTypeDef;

#define RCC_XTH_OFF                             ( 0 )
#define RCC_XTH_ON                              ( RCC_XTHCR_XTHEN )
#define RCC_XTH_BYPASS                          ( RCC_XTHCR_XTHEN | RCC_XTHCR_XTHBYP )

#define RCC_RC64M_DIV1                          ( 0 )
#define RCC_RC64M_DIV16                         ( RCC_RC64MCR_RC64MDIV )

#define RCC_XTL_OFF                             ( 0 )
#define RCC_XTL_ON                              ( RCC_STDBYCTRL_XTLEN )
#define RCC_XTL_BYPASS                          ( RCC_STDBYCTRL_XTLEN | RCC_STDBYCTRL_XTLBYP )

#define RCC_MCO1_POL_LOW                        ( 0 )
#define RCC_MCO1_POL_HIGH                       ( RCC_CLKOCR_BUZZER1EN )

#define RCC_MCO2_POL_LOW                        ( 0 )
#define RCC_MCO2_POL_HIGH                       ( RCC_CLKOCR_BUZZER2EN )


/** @brief  Reset the whole system except for analog IP, EFC, SENSOR and system register module.
  */

#define __RCC_RESET()                           ( RCC->RCR &= ~(RCC_RCR_SRSTMAP) )

/** @brief  Reset the whole system except for analog IP, EFC, SENSOR and system register module, 
  *         but does not change the REMap state.
  */

#define __RCC_RESET_NO_MAP()                    ( RCC->RCR &= ~(RCC_RCR_SRSTNOMAP) )

/** @brief  Flash Reset.
  */

#define __RCC_RESET_EFC()                       ( RCC->RCR &= ~(RCC_RCR_EFCRST) )

/** @brief  Enable or disable the remap.
  */

#define __RCC_REMAP_ENABLE()                    do                                      \
                                                {                                       \
                                                    RCC->RCR &= ~RCC_RCR_REMAPIMP;      \
                                                    RCC->RCR |= RCC_RCR_REMAPIMP;       \
                                                } while (0)
#define __RCC_REMAP_DISABLE()                   ( RCC->RCR &= ~(RCC_RCR_REMAPIMP) )
          
/** @brief  Enable or disable LOCKUP to reset system logic.
  */

#define __RCC_LOCKRSTTOSYSRST_ENABLE()          ( RCC->RCR |= (RCC_RCR_LOCKRSTEN) )
#define __RCC_LOCKRSTTOSYSRST_DISABLE()         ( RCC->RCR &= ~(RCC_RCR_LOCKRSTEN) )
          
/** @brief  Enable or disable IWDT reset to reset system logic.
  */

#define __RCC_IWDTRSTTOSYSRST_ENABLE()          ( RCC->RCR |= (RCC_RCR_IWDTRSTEN) )
#define __RCC_IWDTRSTTOSYSRST_DISABLE()         ( RCC->RCR &= ~(RCC_RCR_IWDTRSTEN) )
          
/** @brief  Enable or disable WDT reset to reset system logic.
  */

#define __RCC_WDTRSTTOSYSRST_ENABLE()           ( RCC->RCR |= (RCC_RCR_WDTRSTEN) )
#define __RCC_WDTRSTTOSYSRST_DISABLE()          ( RCC->RCR &= ~(RCC_RCR_WDTRSTEN) )
          
/** @brief  Enable or disable LVD reset to reset system logic.
  */

#define __RCC_LVDRSTTOSYSRST_ENABLE()           ( RCC->RCR |= (RCC_RCR_LVDRSTEN) )
#define __RCC_LVDRSTTOSYSRST_DISABLE()          ( RCC->RCR &= ~(RCC_RCR_LVDRSTEN) )

/** @defgroup  RCC_Reset_Source RCC Reset Source
  * @{
  */

#define RCC_RESETSOURCE_PWR                     ( RCC_RSR_PWRRSTF )
#define RCC_RESETSOURCE_POR12                   ( RCC_RSR_POR12RSTF )
#define RCC_RESETSOURCE_SOFT                    ( RCC_RSR_SRSTF )
#define RCC_RESETSOURCE_NOMAP                   ( RCC_RSR_SRSTNMF )
#define RCC_RESETSOURCE_EFC                     ( RCC_RSR_EFCRSTF )
#define RCC_RESETSOURCE_RSTN                    ( RCC_RSR_RSTNF )
#define RCC_RESETSOURCE_SYSREQ                  ( RCC_RSR_SYSREQRSTF )
#define RCC_RESETSOURCE_LOCKUP                  ( RCC_RSR_LOCKUPRSTF )
#define RCC_RESETSOURCE_IWDT                    ( RCC_RSR_IWDTRSTF )
#define RCC_RESETSOURCE_WDT                     ( RCC_RSR_WDTRSTF )
#define RCC_RESETSOURCE_LVD                     ( RCC_RSR_LVDRSTF )

/**
  * @}
  */
  
/** @brief Clear Reset Source. 
  */

#define __RCC_CLEAR_RESETSOURCE()               ( RCC->RSR |= RCC_RSR_RSTFLAGCLR )

/** @brief  Check Reset Source. The returned value can be a or more value of @ref RCC_Reset_Source
  */
  
#define __RCC_GET_RESETSOURCE()                 ( RCC->RSR & (RCC_RESETSOURCE_PWR | RCC_RESETSOURCE_POR12 | \
                                                              RCC_RESETSOURCE_SOFT | RCC_RESETSOURCE_NOMAP | \
                                                              RCC_RESETSOURCE_EFC | RCC_RESETSOURCE_RSTN | \
                                                              RCC_RESETSOURCE_SYSREQ | RCC_RESETSOURCE_LOCKUP | \
                                                              RCC_RESETSOURCE_IWDT | RCC_RESETSOURCE_WDT | \
                                                              RCC_RESETSOURCE_LVD))

/******************************************************************************/

/** @defgroup RCC_Sysclk_Source
  */

#define RCC_SYSCLKSOURCE_RC64M                  ( 0U )
#define RCC_SYSCLKSOURCE_RC32K                  ( RCC_CCR1_SYSCLKSEL_0 )
#define RCC_SYSCLKSOURCE_XTH                    ( RCC_CCR1_SYSCLKSEL_1 )
#define RCC_SYSCLKSOURCE_XTL                    ( RCC_CCR1_SYSCLKSEL_1 | RCC_CCR1_SYSCLKSEL_0 )
#define RCC_SYSCLKSOURCE_PLLCLK                 ( RCC_CCR1_SYSCLKSEL_2 )

/**
  * @}
  */

/** @brief RCC sysclk division complete flag. 
  */

#define RCC_FLAG_DIV_DONE                       ( RCC_CCR2_DIVDONE )

/** @defgroup RCC_LPTIM1_CLK_Source
  */

#define RCC_LPTIM1CLKSOURCE_PCLK                ( 0U )
#define RCC_LPTIM1CLKSOURCE_RC32K               ( RCC_CCR2_LPTIM1CLKSEL_0 )
#define RCC_LPTIM1CLKSOURCE_RC64M               ( RCC_CCR2_LPTIM1CLKSEL_1 )
#define RCC_LPTIM1CLKSOURCE_XTL                 ( RCC_CCR2_LPTIM1CLKSEL_1 | RCC_CCR2_LPTIM1CLKSEL_0 )

/**
  * @}
  */

/** @defgroup RCC_FLT_CLK_Source
  */

#define RCC_FLTCLKSOURCE_PCLK_DIV32             ( 0U )
#define RCC_FLTCLKSOURCE_RC32K                  ( RCC_CCR2_FLTCLKSEL )

/**
  * @}
  */

/** @defgroup RCC_LPUART1_CLK_Source
  */

#define RCC_LPUART1CLKSOURCE_RC32K              ( 0U )
#define RCC_LPUART1CLKSOURCE_XTL                ( RCC_CCR2_LPUART1CLKSEL_0 )
#define RCC_LPUART1CLKSOURCE_PCLK_DIV4          ( RCC_CCR2_LPUART1CLKSEL_1 )
#define RCC_LPUART1CLKSOURCE_PCLK_DIV8          ( RCC_CCR2_LPUART1CLKSEL_1 | RCC_CCR2_LPUART1DIV_0 )
#define RCC_LPUART1CLKSOURCE_PCLK_DIV16         ( RCC_CCR2_LPUART1CLKSEL_1 | RCC_CCR2_LPUART1DIV_1 )
#define RCC_LPUART1CLKSOURCE_PCLK_DIV32         ( RCC_CCR2_LPUART1CLKSEL_1 | RCC_CCR2_LPUART1DIV_1 | RCC_CCR2_LPUART1DIV_0 )

/**
  * @}
  */

/** @defgroup RCC_PCLK2_DIV
  */

#define RCC_PCLK2_DIV_1                         ( 0U )
#define RCC_PCLK2_DIV_2                         ( RCC_CCR2_PCLK2DIV_2 )
#define RCC_PCLK2_DIV_4                         ( RCC_CCR2_PCLK2DIV_2 | RCC_CCR2_PCLK2DIV_0)
#define RCC_PCLK2_DIV_8                         ( RCC_CCR2_PCLK2DIV_2 | RCC_CCR2_PCLK2DIV_1 )
#define RCC_PCLK2_DIV_16                        ( RCC_CCR2_PCLK2DIV_2 | RCC_CCR2_PCLK2DIV_1 | RCC_CCR2_PCLK2DIV_0 )

/**
  * @}
  */

/** @defgroup RCC_PCLK1_DIV
  */

#define RCC_PCLK1_DIV_1                         ( 0U )
#define RCC_PCLK1_DIV_2                         ( RCC_CCR2_PCLK1DIV_2 )
#define RCC_PCLK1_DIV_4                         ( RCC_CCR2_PCLK1DIV_2 | RCC_CCR2_PCLK1DIV_0)
#define RCC_PCLK1_DIV_8                         ( RCC_CCR2_PCLK1DIV_2 | RCC_CCR2_PCLK1DIV_1 )
#define RCC_PCLK1_DIV_16                        ( RCC_CCR2_PCLK1DIV_2 | RCC_CCR2_PCLK1DIV_1 | RCC_CCR2_PCLK1DIV_0 )

/**
  * @}
  */

/** @defgroup RCC_Sysclk_DIV1
  */

#define RCC_SYSDIV1_DIV_1                       ( 0U )
#define RCC_SYSDIV1_DIV_2                       ( RCC_CCR2_SYSDIV1_0 )
#define RCC_SYSDIV1_DIV_3                       ( RCC_CCR2_SYSDIV1_1 )
#define RCC_SYSDIV1_DIV_4                       ( RCC_CCR2_SYSDIV1_1 | RCC_CCR2_SYSDIV1_0 )
#define RCC_SYSDIV1_DIV_5                       ( RCC_CCR2_SYSDIV1_2 )
#define RCC_SYSDIV1_DIV_6                       ( RCC_CCR2_SYSDIV1_2 | RCC_CCR2_SYSDIV1_0 )
#define RCC_SYSDIV1_DIV_7                       ( RCC_CCR2_SYSDIV1_2 | RCC_CCR2_SYSDIV1_1 )
#define RCC_SYSDIV1_DIV_8                       ( RCC_CCR2_SYSDIV1_2 | RCC_CCR2_SYSDIV1_1 | RCC_CCR2_SYSDIV1_0 )
#define RCC_SYSDIV1_DIV_9                       ( RCC_CCR2_SYSDIV1_3 )
#define RCC_SYSDIV1_DIV_10                      ( RCC_CCR2_SYSDIV1_3 | RCC_CCR2_SYSDIV1_0 )
#define RCC_SYSDIV1_DIV_11                      ( RCC_CCR2_SYSDIV1_3 | RCC_CCR2_SYSDIV1_1 )
#define RCC_SYSDIV1_DIV_12                      ( RCC_CCR2_SYSDIV1_3 | RCC_CCR2_SYSDIV1_1 | RCC_CCR2_SYSDIV1_0 )
#define RCC_SYSDIV1_DIV_13                      ( RCC_CCR2_SYSDIV1_3 | RCC_CCR2_SYSDIV1_2 )
#define RCC_SYSDIV1_DIV_14                      ( RCC_CCR2_SYSDIV1_3 | RCC_CCR2_SYSDIV1_2 | RCC_CCR2_SYSDIV1_0 )
#define RCC_SYSDIV1_DIV_15                      ( RCC_CCR2_SYSDIV1_3 | RCC_CCR2_SYSDIV1_2 | RCC_CCR2_SYSDIV1_1 )
#define RCC_SYSDIV1_DIV_16                      ( RCC_CCR2_SYSDIV1_3 | RCC_CCR2_SYSDIV1_2 | RCC_CCR2_SYSDIV1_1 | RCC_CCR2_SYSDIV1_0 )

/**
  * @}
  */

/** @defgroup RCC_Sysclk_DIV0
  */

#define RCC_SYSDIV0_DIV_1                       ( 0U )
#define RCC_SYSDIV0_DIV_2                       ( RCC_CCR2_SYSDIV0_0 )
#define RCC_SYSDIV0_DIV_3                       ( RCC_CCR2_SYSDIV0_1 )
#define RCC_SYSDIV0_DIV_4                       ( RCC_CCR2_SYSDIV0_1 | RCC_CCR2_SYSDIV0_0 )
#define RCC_SYSDIV0_DIV_5                       ( RCC_CCR2_SYSDIV0_2 )
#define RCC_SYSDIV0_DIV_6                       ( RCC_CCR2_SYSDIV0_2 | RCC_CCR2_SYSDIV0_0 )
#define RCC_SYSDIV0_DIV_7                       ( RCC_CCR2_SYSDIV0_2 | RCC_CCR2_SYSDIV0_1 )
#define RCC_SYSDIV0_DIV_8                       ( RCC_CCR2_SYSDIV0_2 | RCC_CCR2_SYSDIV0_1 | RCC_CCR2_SYSDIV0_0 )
#define RCC_SYSDIV0_DIV_9                       ( RCC_CCR2_SYSDIV0_3 )
#define RCC_SYSDIV0_DIV_10                      ( RCC_CCR2_SYSDIV0_3 | RCC_CCR2_SYSDIV0_0 )
#define RCC_SYSDIV0_DIV_11                      ( RCC_CCR2_SYSDIV0_3 | RCC_CCR2_SYSDIV0_1 )
#define RCC_SYSDIV0_DIV_12                      ( RCC_CCR2_SYSDIV0_3 | RCC_CCR2_SYSDIV0_1 | RCC_CCR2_SYSDIV0_0 )
#define RCC_SYSDIV0_DIV_13                      ( RCC_CCR2_SYSDIV0_3 | RCC_CCR2_SYSDIV0_2 )
#define RCC_SYSDIV0_DIV_14                      ( RCC_CCR2_SYSDIV0_3 | RCC_CCR2_SYSDIV0_2 | RCC_CCR2_SYSDIV0_0 )
#define RCC_SYSDIV0_DIV_15                      ( RCC_CCR2_SYSDIV0_3 | RCC_CCR2_SYSDIV0_2 | RCC_CCR2_SYSDIV0_1 )
#define RCC_SYSDIV0_DIV_16                      ( RCC_CCR2_SYSDIV0_3 | RCC_CCR2_SYSDIV0_2 | RCC_CCR2_SYSDIV0_1 | RCC_CCR2_SYSDIV0_0 )

/**
  * @}
  */

/** @defgroup RCC_Interrupt
  */

#define RCC_IT_RC32K                            ( RCC_CIR_RC32KRDYIE )
#define RCC_IT_XTL                              ( RCC_CIR_XTLRDYIE )
#define RCC_IT_RC64M                            ( RCC_CIR_RC64MRDYIE )
#define RCC_IT_XTH                              ( RCC_CIR_XTHRDYIE )
#define RCC_IT_PLL                              ( RCC_CIR_PLLLOCKIE )

/**
  * @}
  */

/** @defgroup RCC_Interrupt_Flag
  */

#define RCC_IT_FLAG_RC32K                       ( RCC_CIR_RC32KRDYIF )
#define RCC_IT_FLAG_XTL                         ( RCC_CIR_XTLRDYIF )
#define RCC_IT_FLAG_RC64M                       ( RCC_CIR_RC64MRDYIF )
#define RCC_IT_FLAG_XTH                         ( RCC_CIR_XTHRDYIF )
#define RCC_IT_FLAG_PLL                         ( RCC_CIR_PLLLOCKIF )

/**
  * @}
  */

/** @brief  Enable or disable Interrupt. This parameter can be a or more value of @ref RCC_Interrupt.
  */

#define __RCC_ENABLE_IT(__INTERRUPT__)          ( RCC->CIR |= (__INTERRUPT__) )
#define __RCC_DISABLE_IT(__INTERRUPT__)         ( RCC->CIR &= ~(__INTERRUPT__) )

/** @brief  Check Interrupt. The returned value can be a or more value of @ref RCC_Interrupt
  */

#define __RCC_GET_IT(__INTERRUPT__)             ( RCC->CIR & ((__INTERRUPT__) >> 8) )

/** @brief  Clear Interrupt Flag. This parameter can be a or more value of @ref RCC_Interrupt.
  */

#define __RCC_CLEAR_IT(__INTERRUPT__)           ( RCC->CIR |= ((__INTERRUPT__) << 8) )

/** @brief  RC64M 16 frequency division.
  */

#define RCC_RC64M_DIV_1                         ( 0U )
#define RCC_RC64M_DIV_16                        ( RCC_RC64MCR_RC64MDIV )

/** @brief  Enable or disable RC64M.
  */

#define __RCC_RC64M_ENABLE()                    ( RCC->RC64MCR |= RCC_RC64MCR_RC64MEN )
#define __RCC_RC64M_DISABLE()                   ( RCC->RC64MCR &= ~RCC_RC64MCR_RC64MEN )

/** @brief  Check whether the RC64M clock is stable.
  */

#define __RCC_RC64M_ISREADY()                   ( (RCC->RC64MCR & RCC_RC64MCR_RC64MRDY) != 0U )

/** @brief  The RC64M clock stable time.
  */

#define RCC_XTH_STABILITY_1024                  ( 0U )
#define RCC_XTH_STABILITY_4096                  ( RCC_XTHCR_XTHRDYTIME_0 )
#define RCC_XTH_STABILITY_16384                 ( RCC_XTHCR_XTHRDYTIME_1 )
#define RCC_XTH_STABILITY_32768                 ( RCC_XTHCR_XTHRDYTIME_1 | RCC_XTHCR_XTHRDYTIME_0 )

/** @brief  Enable or disable XTH bypass mode.
  */

#define __RCC_XTH_BYPASS_ENABLE()               ( RCC->XTHCR |= RCC_XTHCR_XTHBYP )
#define __RCC_XTH_BYPASS_DISABLE()              ( RCC->XTHCR &= ~RCC_XTHCR_XTHBYP )

/** @brief  Enable or disable XTH.
  */

#define __RCC_XTH_ENABLE()                      ( RCC->XTHCR |= RCC_XTHCR_XTHEN )
#define __RCC_XTH_DISABLE()                     ( RCC->XTHCR &= ~RCC_XTHCR_XTHEN )

/** @brief  Check whether the XTH clock is stable.
  */

#define __RCC_XTH_ISREADY()                     ( (RCC->XTHCR & RCC_XTHCR_XTHRDY) != 0U )

/** @brief  RCC PLL Lock Source.
  */

#define RCC_PLLLOCKSOURCE_LOCK                  ( 0U )
#define RCC_PLLLOCKSOURCE_FREE                  ( RCC_PLLCR_PLLLOCKSEL )

/** @defgroup RCC_PLL_Source
  */

#define RCC_PLLSOURCE_RC64M_DIV16               ( 0U )
#define RCC_PLLSOURCE_XTH                       ( RCC_PLLCR_PLLSRCSEL_1 )

/**
  * @}
  */

/** @brief  Check whether the PLL clock is stable.
  */

#define __RCC_PLL_ISREADY()                     ( RCC->PLLCR & RCC_PLLCR_PLLLOCK )

/** @brief  The PLL configuration updates the control bit. 
  * @note   Rewrite PLL_MPLL_N and PLL_FPLL_SRC After SEL, 
  *         this bit needs to be set to a new configuration to take effect.
  */

#define __RCC_PLLUPDATE()                       ( RCC->PLLCR |= RCC_PLLCR_PLLUPDATEEN )

/** @brief  Enable or disable PLL sleep.
  */

#define __RCC_PLLSLEEP_ENABLE()                 ( RCC->PLLCR |= RCC_PLLCR_PLLSLEEP )
#define __RCC_PLLSLEEP_DISABLE()                ( RCC->PLLCR &= ~RCC_PLLCR_PLLSLEEP )

/** @brief  Enable or disable PLL.
  */

#define __RCC_PLL_ENABLE()                      ( RCC->PLLCR |= RCC_PLLCR_PLLEN )
#define __RCC_PLL_DISABLE()                     ( RCC->PLLCR &= ~RCC_PLLCR_PLLEN )

/** @defgroup RCC_MCO_Output
  */

#define RCC_MCO_NONE                            ( RCC_CLKOCR_CLKTESTSEL_3 | RCC_CLKOCR_CLKTESTSEL_2 | RCC_CLKOCR_CLKTESTSEL_1 | RCC_CLKOCR_CLKTESTSEL_0 )
#define RCC_MCO_HCLK                            ( 0U )
#define RCC_MCO_RC64M                           ( RCC_CLKOCR_CLKTESTSEL_0 )
#define RCC_MCO_RC32K                           ( RCC_CLKOCR_CLKTESTSEL_1 )
#define RCC_MCO_XTH                             ( RCC_CLKOCR_CLKTESTSEL_1 | RCC_CLKOCR_CLKTESTSEL_0 )
#define RCC_MCO_XTL                             ( RCC_CLKOCR_CLKTESTSEL_2 )
#define RCC_MCO_PLLCLK                          ( RCC_CLKOCR_CLKTESTSEL_2 | RCC_CLKOCR_CLKTESTSEL_0 )
#define RCC_MCO_PCLK1                           ( RCC_CLKOCR_CLKTESTSEL_2 | RCC_CLKOCR_CLKTESTSEL_1 )
#define RCC_MCO_PCLK2                           ( RCC_CLKOCR_CLKTESTSEL_2 | RCC_CLKOCR_CLKTESTSEL_1 | RCC_CLKOCR_CLKTESTSEL_0 )
#define RCC_MCO_SYSCLK                          ( RCC_CLKOCR_CLKTESTSEL_3 )
#define RCC_MCO_TIMCLK1                         ( RCC_CLKOCR_CLKTESTSEL_3 | RCC_CLKOCR_CLKTESTSEL_0 )
#define RCC_MCO_TIMCLK2                         ( RCC_CLKOCR_CLKTESTSEL_3 | RCC_CLKOCR_CLKTESTSEL_1 )
#define RCC_MCO_LPUART1CLK                      ( RCC_CLKOCR_CLKTESTSEL_3 | RCC_CLKOCR_CLKTESTSEL_1 | RCC_CLKOCR_CLKTESTSEL_0 )
#define RCC_MCO_FCLKDIV8                        ( RCC_CLKOCR_CLKTESTSEL_3 | RCC_CLKOCR_CLKTESTSEL_2 )
#define RCC_MCO_USBPHYPLL48M                    ( RCC_CLKOCR_CLKTESTSEL_3 | RCC_CLKOCR_CLKTESTSEL_2 | RCC_CLKOCR_CLKTESTSEL_0 )
#define RCC_MCO_RTC_PCLK                        ( RCC_CLKOCR_CLKTESTSEL_3 | RCC_CLKOCR_CLKTESTSEL_2 | RCC_CLKOCR_CLKTESTSEL_1 )

/**
  * @}
  */

/** @brief  Check whether the XTL clock is stable.
  */

#define __RCC_XTL_ISREADY()                     ( (RCC->STDBYCTRL & RCC_STDBYCTRL_XTLRDY) != 0U )

/** @brief  Check whether the RC32K clock is stable.
  */

#define __RCC_RC32K_ISREADY()                   ( (RCC->STDBYCTRL & RCC_STDBYCTRL_RC32KRDY) != 0U )

/** @brief  Enable or disable RC32K.
  */

#define __RCC_RC32K_ENABLE()                    ( RCC->STDBYCTRL |= RCC_STDBYCTRL_RC32EN )
#define __RCC_RC32K_DISABLE()                   ( RCC->STDBYCTRL &= ~RCC_STDBYCTRL_RC32EN )

/** @defgroup RCC_RTC_Clock_Source
  */

#define RCC_RTCCLKSOURCE_RC32K                  ( 0U )
#define RCC_RTCCLKSOURCE_XTL                    ( RCC_STDBYCTRL_RTCSEL_0 )
#define RCC_RTCCLKSOURCE_NONE                   ( RCC_STDBYCTRL_RTCSEL_1 )

/**
  * @}
  */

/** @defgroup RCC_XTL_Drive_Capacity.
  */

#define RCC_XTL_DRIVE0                          ( 0U )
#define RCC_XTL_DRIVE1                          ( RCC_STDBYCTRL_XTLDRV_0 )
#define RCC_XTL_DRIVE2                          ( RCC_STDBYCTRL_XTLDRV_1 )
#define RCC_XTL_DRIVE3                          ( RCC_STDBYCTRL_XTLDRV_1 | RCC_STDBYCTRL_XTLDRV_0 )
#define RCC_XTL_LOWPOWER_DRIVE0                 ( RCC_STDBYCTRL_XTLDRV_2 )
#define RCC_XTL_LOWPOWER_DRIVE1                 ( RCC_STDBYCTRL_XTLDRV_2 | RCC_STDBYCTRL_XTLDRV_0 )
#define RCC_XTL_LOWPOWER_DRIVE2                 ( RCC_STDBYCTRL_XTLDRV_2 | RCC_STDBYCTRL_XTLDRV_1 )
#define RCC_XTL_LOWPOWER_DRIVE3                 ( RCC_STDBYCTRL_XTLDRV_2 | RCC_STDBYCTRL_XTLDRV_1 | RCC_STDBYCTRL_XTLDRV_0 )

/**
  * @}
  */

/** @brief  the APB1 peripheral reset.
  */

#define __RCC_LPUART1_RESET()                   do                                              \
                                                {                                               \
                                                    RCC->APB1RSTR &= ~RCC_APB1RSTR_LPUART1RST;  \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->APB1RSTR |= RCC_APB1RSTR_LPUART1RST;   \
                                                }while (0)
                                                
#define __RCC_LPTIM1_RESET()                    do                                              \
                                                {                                               \
                                                    RCC->APB1RSTR &= ~RCC_APB1RSTR_LPTIM1RST;   \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->APB1RSTR |= RCC_APB1RSTR_LPTIM1RST;    \
                                                }while (0)
                                                
#define __RCC_PMU_RESET()                       do                                              \
                                                {                                               \
                                                    RCC->APB1RSTR &= ~RCC_APB1RSTR_PMURST;      \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->APB1RSTR |= RCC_APB1RSTR_PMURST;       \
                                                }while (0)
                                                
#define __RCC_CAN2_RESET()                      do                                              \
                                                {                                               \
                                                    RCC->APB1RSTR &= ~RCC_APB1RSTR_CAN2RST;     \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->APB1RSTR |= RCC_APB1RSTR_CAN2RST;      \
                                                }while (0)
                                                
#define __RCC_CAN1_RESET()                      do                                              \
                                                {                                               \
                                                    RCC->APB1RSTR &= ~RCC_APB1RSTR_CAN1RST;     \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->APB1RSTR |= RCC_APB1RSTR_CAN1RST;      \
                                                }while (0)
                                                
#define __RCC_I2C2_RESET()                      do                                              \
                                                {                                               \
                                                    RCC->APB1RSTR &= ~RCC_APB1RSTR_I2C2RST;     \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->APB1RSTR |= RCC_APB1RSTR_I2C2RST;      \
                                                }while (0)
                                                
#define __RCC_I2C1_RESET()                      do                                              \
                                                {                                               \
                                                    RCC->APB1RSTR &= ~RCC_APB1RSTR_I2C1RST;     \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->APB1RSTR |= RCC_APB1RSTR_I2C1RST;      \
                                                }while (0)
                                                
#define __RCC_UART4_RESET()                     do                                              \
                                                {                                               \
                                                    RCC->APB1RSTR &= ~RCC_APB1RSTR_UART4RST;    \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->APB1RSTR |= RCC_APB1RSTR_UART4RST;     \
                                                }while (0)
                                                
#define __RCC_UART3_RESET()                     do                                              \
                                                {                                               \
                                                    RCC->APB1RSTR &= ~RCC_APB1RSTR_UART3RST;    \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->APB1RSTR |= RCC_APB1RSTR_UART3RST;     \
                                                }while (0)
                                                
#define __RCC_UART2_RESET()                     do                                              \
                                                {                                               \
                                                    RCC->APB1RSTR &= ~RCC_APB1RSTR_UART2RST;    \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->APB1RSTR |= RCC_APB1RSTR_UART2RST;     \
                                                }while (0)
                                                
#define __RCC_I2S2_RESET()                      do                                              \
                                                {                                               \
                                                    RCC->APB1RSTR &= ~RCC_APB1RSTR_I2S2RST;     \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->APB1RSTR |= RCC_APB1RSTR_I2S2RST;      \
                                                }while (0)
                                                
#define __RCC_I2S1_RESET()                      do                                              \
                                                {                                               \
                                                    RCC->APB1RSTR &= ~RCC_APB1RSTR_I2S1RST;     \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->APB1RSTR |= RCC_APB1RSTR_I2S1RST;      \
                                                }while (0)
                                                
#define __RCC_WDT_RESET()                       do                                              \
                                                {                                               \
                                                    RCC->APB1RSTR &= ~RCC_APB1RSTR_WDTRST;      \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->APB1RSTR |= RCC_APB1RSTR_WDTRST;       \
                                                }while (0)
                                                
#define __RCC_RTC_RESET()                       do                                              \
                                                {                                               \
                                                    RCC->APB1RSTR &= ~RCC_APB1RSTR_RTCRST;      \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->APB1RSTR |= RCC_APB1RSTR_RTCRST;       \
                                                }while (0)
                                                
#define __RCC_TIM7_RESET()                      do                                              \
                                                {                                               \
                                                    RCC->APB1RSTR &= ~RCC_APB1RSTR_TIM7RST;     \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->APB1RSTR |= RCC_APB1RSTR_TIM7RST;      \
                                                }while (0)
                                                
#define __RCC_TIM6_RESET()                      do                                              \
                                                {                                               \
                                                    RCC->APB1RSTR &= ~RCC_APB1RSTR_TIM6RST;     \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->APB1RSTR |= RCC_APB1RSTR_TIM6RST;      \
                                                }while (0)
                                                
#define __RCC_TIM4_RESET()                      do                                              \
                                                {                                               \
                                                    RCC->APB1RSTR &= ~RCC_APB1RSTR_TIM4RST;     \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->APB1RSTR |= RCC_APB1RSTR_TIM4RST;      \
                                                }while (0)
                                                
#define __RCC_TIM3_RESET()                      do                                              \
                                                {                                               \
                                                    RCC->APB1RSTR &= ~RCC_APB1RSTR_TIM3RST;     \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->APB1RSTR |= RCC_APB1RSTR_TIM3RST;      \
                                                }while (0)
                                                
#define __RCC_TIM2_RESET()                      do                                              \
                                                {                                               \
                                                    RCC->APB1RSTR &= ~RCC_APB1RSTR_TIM2RST;     \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->APB1RSTR |= RCC_APB1RSTR_TIM2RST;      \
                                                }while (0)

/** @brief  The APB2 peripheral reset.
  */

#define __RCC_TIM17_RESET()                     do                                              \
                                                {                                               \
                                                    RCC->APB2RSTR &= ~RCC_APB2RSTR_TIM17RST;    \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->APB2RSTR |= RCC_APB2RSTR_TIM17RST;     \
                                                }while (0)
                                                
#define __RCC_TIM16_RESET()                     do                                              \
                                                {                                               \
                                                    RCC->APB2RSTR &= ~RCC_APB2RSTR_TIM16RST;    \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->APB2RSTR |= RCC_APB2RSTR_TIM16RST;     \
                                                }while (0)
                                                
#define __RCC_TIM15_RESET()                     do                                              \
                                                {                                               \
                                                    RCC->APB2RSTR &= ~RCC_APB2RSTR_TIM15RST;    \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->APB2RSTR |= RCC_APB2RSTR_TIM15RST;     \
                                                }while (0)
                                                
#define __RCC_UART1_RESET()                     do                                              \
                                                {                                               \
                                                    RCC->APB2RSTR &= ~RCC_APB2RSTR_UART1RST;    \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->APB2RSTR |= RCC_APB2RSTR_UART1RST;     \
                                                }while (0)
                                                
#define __RCC_TIM8_RESET()                      do                                              \
                                                {                                               \
                                                    RCC->APB2RSTR &= ~RCC_APB2RSTR_TIM8RST;     \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->APB2RSTR |= RCC_APB2RSTR_TIM8RST;      \
                                                }while (0)
                                                
#define __RCC_TIM1_RESET()                      do                                              \
                                                {                                               \
                                                    RCC->APB2RSTR &= ~RCC_APB2RSTR_TIM1RST;     \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->APB2RSTR |= RCC_APB2RSTR_TIM1RST;      \
                                                }while (0)
                                                
#define __RCC_EXTI_RESET()                      do                                              \
                                                {                                               \
                                                    RCC->APB2RSTR &= ~RCC_APB2RSTR_EXTIRST;     \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->APB2RSTR |= RCC_APB2RSTR_EXTIRST;      \
                                                }while (0)
                                                
#define __RCC_OPA_RESET()                       do                                              \
                                                {                                               \
                                                    RCC->APB2RSTR &= ~RCC_APB2RSTR_OPARST;      \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->APB2RSTR |= RCC_APB2RSTR_OPARST;       \
                                                }while (0)
                                                
#define __RCC_COMP_RESET()                      do                                             \
                                                {                                               \
                                                    RCC->APB2RSTR &= ~RCC_APB2RSTR_COMPRST;     \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->APB2RSTR |= RCC_APB2RSTR_COMPRST;      \
                                                }while (0)
                                                
#define __RCC_VREFBUF_RESET()                   do                                              \
                                                {                                               \
                                                    RCC->APB2RSTR &= ~RCC_APB2RSTR_VREFBUFRST;  \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->APB2RSTR |= RCC_APB2RSTR_VREFBUFRST;   \
                                                }while (0)
                                                
#define __RCC_SYSCFG_RESET()                    do                                              \
                                                {                                               \
                                                    RCC->APB2RSTR &= ~RCC_APB2RSTR_SYSCFGRST;   \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->APB2RSTR |= RCC_APB2RSTR_SYSCFGRST;    \
                                                }while (0)
                                                
/** @brief  The AHB peripheral reset.
  */

#define __RCC_EXMC_RESET()                      do                                              \
                                                {                                               \
                                                    RCC->AHBRSTR &= ~RCC_AHBRSTR_EXMCRST;        \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->AHBRSTR |= RCC_AHBRSTR_EXMCRST;         \
                                                }while (0)

#define __RCC_ROM_RESET()                       do                                              \
                                                {                                               \
                                                    RCC->AHBRSTR &= ~RCC_AHBRSTR_ROMRST;        \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->AHBRSTR |= RCC_AHBRSTR_ROMRST;         \
                                                }while (0)

#define __RCC_FAU_RESET()                       do                                              \
                                                {                                               \
                                                    RCC->AHBRSTR &= ~RCC_AHBRSTR_FAURST;        \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->AHBRSTR |= RCC_AHBRSTR_FAURST;         \
                                                }while (0)

#define __RCC_HRNG_RESET()                      do                                              \
                                                {                                               \
                                                    RCC->AHBRSTR &= ~RCC_AHBRSTR_HRNGRST;       \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->AHBRSTR |= RCC_AHBRSTR_HRNGRST;        \
                                                }while (0)

#define __RCC_AES_RESET()                       do                                              \
                                                {                                               \
                                                    RCC->AHBRSTR &= ~RCC_AHBRSTR_AESRST;        \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->AHBRSTR |= RCC_AHBRSTR_AESRST;         \
                                                }while (0)

#define __RCC_DAC1_RESET()                      do                                              \
                                                {                                               \
                                                    RCC->AHBRSTR &= ~RCC_AHBRSTR_DAC1RST;       \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->AHBRSTR |= RCC_AHBRSTR_DAC1RST;        \
                                                }while (0)

#define __RCC_ADC12_RESET()                     do                                              \
                                                {                                               \
                                                    RCC->AHBRSTR &= ~RCC_AHBRSTR_ADC12RST;      \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->AHBRSTR |= RCC_AHBRSTR_ADC12RST;       \
                                                }while (0)

#define __RCC_GPIOF_RESET()                     do                                              \
                                                {                                               \
                                                    RCC->AHBRSTR &= ~RCC_AHBRSTR_GPIOFRST;      \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->AHBRSTR |= RCC_AHBRSTR_GPIOFRST;       \
                                                }while (0)

#define __RCC_GPIOE_RESET()                     do                                              \
                                                {                                               \
                                                    RCC->AHBRSTR &= ~RCC_AHBRSTR_GPIOERST;      \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->AHBRSTR |= RCC_AHBRSTR_GPIOERST )      \
                                                }while (0)

#define __RCC_GPIOD_RESET()                     do                                              \
                                                {                                               \
                                                    RCC->AHBRSTR &= ~RCC_AHBRSTR_GPIODRST;      \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->AHBRSTR |= RCC_AHBRSTR_GPIODRST;       \
                                                }while (0)

#define __RCC_GPIOC_RESET()                     do                                              \
                                                {                                               \
                                                    RCC->AHBRSTR &= ~RCC_AHBRSTR_GPIOCRST;      \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->AHBRSTR |= RCC_AHBRSTR_GPIOCRST;       \
                                                }while (0)

#define __RCC_GPIOB_RESET()                     do                                              \
                                                {                                               \
                                                    RCC->AHBRSTR &= ~RCC_AHBRSTR_GPIOBRST;      \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->AHBRSTR |= RCC_AHBRSTR_GPIOBRST;       \
                                                }while (0)

#define __RCC_GPIOA_RESET()                     do                                              \
                                                {                                               \
                                                    RCC->AHBRSTR &= ~RCC_AHBRSTR_GPIOARST;      \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->AHBRSTR |= RCC_AHBRSTR_GPIOARST;       \
                                                }while (0)

#define __RCC_SPI3_RESET()                      do                                              \
                                                {                                               \
                                                    RCC->AHBRSTR &= ~RCC_AHBRSTR_SPI3RST;       \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->AHBRSTR |= RCC_AHBRSTR_SPI3RST;        \
                                                }while (0)

#define __RCC_SPI2_RESET()                      do                                               \
                                                {                                               \
                                                    RCC->AHBRSTR &= ~RCC_AHBRSTR_SPI2RST;       \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->AHBRSTR |= RCC_AHBRSTR_SPI2RST;        \
                                                }while (0)

#define __RCC_SPI1_RESET()                      do                                              \
                                                {                                               \
                                                    RCC->AHBRSTR &= ~RCC_AHBRSTR_SPI1RST;       \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->AHBRSTR |= RCC_AHBRSTR_SPI1RST;        \
                                                }while (0)

#define __RCC_CRC_RESET()                       do                                              \
                                                {                                               \
                                                    RCC->AHBRSTR &= ~RCC_AHBRSTR_CRCRST;        \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->AHBRSTR |= RCC_AHBRSTR_CRCRST;         \
                                                }while (0)

#define __RCC_USB_RESET()                       do                                              \
                                                {                                               \
                                                    RCC->AHBRSTR &= ~RCC_AHBRSTR_USBRST;        \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->AHBRSTR |= RCC_AHBRSTR_USBRST;         \
                                                }while (0)

#define __RCC_EFC_RESET()                       do                                              \
                                                {                                               \
                                                    RCC->AHBRSTR &= ~RCC_AHBRSTR_EFCRST;        \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->AHBRSTR |= RCC_AHBRSTR_EFCRST;         \
                                                }while (0)

#define __RCC_SRAM_RESET()                      do                                              \
                                                {                                               \
                                                    RCC->AHBRSTR &= ~RCC_AHBRSTR_SRAMRST;       \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->AHBRSTR |= RCC_AHBRSTR_SRAMRST;        \
                                                }while (0)

#define __RCC_DMA_RESET()                       do                                              \
                                                {                                               \
                                                    RCC->AHBRSTR &= ~RCC_AHBRSTR_DMARST;       \
                                                    __NOP();__NOP();__NOP();                    \
                                                    RCC->AHBRSTR |= RCC_AHBRSTR_DMARST;        \
                                                }while (0)

/** @brief  Enable or disable the APB1 peripheral clock.
  */

#define __RCC_LPUART1_CLK_ENABLE()              do                                              \
                                                {                                               \
                                                    RCC->APB1ENR |= RCC_APB1ENR_LPUART1CKEN;    \
                                                    RCC->APB2ENR |= RCC_APB2ENR_EXTICKEN;       \
                                                }while (0)
#define __RCC_LPTIM1_CLK_ENABLE()               do                                              \
                                                {                                               \
                                                    RCC->APB1ENR |= RCC_APB1ENR_LPTIM1CKEN;     \
                                                    RCC->APB2ENR |= (RCC_APB2ENR_EXTICKEN);     \
                                                }while (0)
#define __RCC_PMU_CLK_ENABLE()                  do                                              \
                                                {                                               \
                                                    RCC->APB1ENR |= RCC_APB1ENR_PMUCKEN;    \
                                                    RCC->APB2ENR |= RCC_APB2ENR_EXTICKEN;       \
                                                }while (0)
#define __RCC_CAN2_CLK_ENABLE()                 ( RCC->APB1ENR |= (RCC_APB1ENR_CAN2CKEN) )
#define __RCC_CAN1_CLK_ENABLE()                 ( RCC->APB1ENR |= (RCC_APB1ENR_CAN1CKEN) )
#define __RCC_I2C2_CLK_ENABLE()                 ( RCC->APB1ENR |= (RCC_APB1ENR_I2C2CKEN) )
#define __RCC_I2C1_CLK_ENABLE()                 ( RCC->APB1ENR |= (RCC_APB1ENR_I2C1CKEN) )
#define __RCC_UART4_CLK_ENABLE()                ( RCC->APB1ENR |= (RCC_APB1ENR_UART4CKEN) )
#define __RCC_UART3_CLK_ENABLE()                ( RCC->APB1ENR |= (RCC_APB1ENR_UART3CKEN) )
#define __RCC_UART2_CLK_ENABLE()                ( RCC->APB1ENR |= (RCC_APB1ENR_UART2CKEN) )
#define __RCC_I2S2_CLK_ENABLE()                 ( RCC->APB1ENR |= (RCC_APB1ENR_I2S2CKEN) )
#define __RCC_I2S1_CLK_ENABLE()                 ( RCC->APB1ENR |= (RCC_APB1ENR_I2S1CKEN) )
#define __RCC_WDT_CLK_ENABLE()                  ( RCC->APB1ENR |= (RCC_APB1ENR_WDTCKEN) )
#define __RCC_RTC_CLK_ENABLE()                  do                                              \
                                                {                                               \
                                                    RCC->STDBYCTRL |= RCC_STDBYCTRL_RTCEN;      \
                                                    RCC->APB1ENR |= RCC_APB1ENR_RTCCKEN;        \
                                                    RCC->APB2ENR |= (RCC_APB2ENR_EXTICKEN);     \
                                                }while (0)
#define __RCC_TIM7_CLK_ENABLE()                 ( RCC->APB1ENR |= (RCC_APB1ENR_TIM7CKEN) )
#define __RCC_TIM6_CLK_ENABLE()                 ( RCC->APB1ENR |= (RCC_APB1ENR_TIM6CKEN) )
#define __RCC_TIM4_CLK_ENABLE()                 ( RCC->APB1ENR |= (RCC_APB1ENR_TIM4CKEN) )
#define __RCC_TIM3_CLK_ENABLE()                 ( RCC->APB1ENR |= (RCC_APB1ENR_TIM3CKEN) )
#define __RCC_TIM2_CLK_ENABLE()                 ( RCC->APB1ENR |= (RCC_APB1ENR_TIM2CKEN) )

#define __RCC_LPUART1_CLK_DISABLE()             ( RCC->APB1ENR &= ~(RCC_APB1ENR_LPUART1CKEN) )
#define __RCC_LPTIM1_CLK_DISABLE()              ( RCC->APB1ENR &= ~(RCC_APB1ENR_LPTIM1CKEN) )
#define __RCC_PMU_CLK_DISABLE()                 ( RCC->APB1ENR &= ~(RCC_APB1ENR_PMUCKEN) )
#define __RCC_CAN2_CLK_DISABLE()                ( RCC->APB1ENR &= ~(RCC_APB1ENR_CAN2CKEN) )
#define __RCC_CAN1_CLK_DISABLE()                ( RCC->APB1ENR &= ~(RCC_APB1ENR_CAN1CKEN) )
#define __RCC_I2C2_CLK_DISABLE()                ( RCC->APB1ENR &= ~(RCC_APB1ENR_I2C2CKEN) )
#define __RCC_I2C1_CLK_DISABLE()                ( RCC->APB1ENR &= ~(RCC_APB1ENR_I2C1CKEN) )
#define __RCC_UART4_CLK_DISABLE()               ( RCC->APB1ENR &= ~(RCC_APB1ENR_UART4CKEN) )
#define __RCC_UART3_CLK_DISABLE()               ( RCC->APB1ENR &= ~(RCC_APB1ENR_UART3CKEN) )
#define __RCC_UART2_CLK_DISABLE()               ( RCC->APB1ENR &= ~(RCC_APB1ENR_UART2CKEN) )
#define __RCC_I2S2_CLK_DISABLE()                ( RCC->APB1ENR &= ~(RCC_APB1ENR_I2S2CKEN) )
#define __RCC_I2S1_CLK_DISABLE()                ( RCC->APB1ENR &= ~(RCC_APB1ENR_I2S1CKEN) )
#define __RCC_WDT_CLK_DISABLE()                 ( RCC->APB1ENR &= ~(RCC_APB1ENR_WDTCKEN) )
#define __RCC_RTC_CLK_DISABLE()                 do                                              \
                                                {                                               \
                                                    RCC->APB1ENR &= ~RCC_APB1ENR_RTCCKEN;       \
                                                    RCC->STDBYCTRL &= ~RCC_STDBYCTRL_RTCEN;     \
                                                }while (0)
#define __RCC_TIM7_CLK_DISABLE()                ( RCC->APB1ENR &= ~(RCC_APB1ENR_TIM7CKEN) )
#define __RCC_TIM6_CLK_DISABLE()                ( RCC->APB1ENR &= ~(RCC_APB1ENR_TIM6CKEN) )
#define __RCC_TIM4_CLK_DISABLE()                ( RCC->APB1ENR &= ~(RCC_APB1ENR_TIM4CKEN) )
#define __RCC_TIM3_CLK_DISABLE()                ( RCC->APB1ENR &= ~(RCC_APB1ENR_TIM3CKEN) )
#define __RCC_TIM2_CLK_DISABLE()                ( RCC->APB1ENR &= ~(RCC_APB1ENR_TIM2CKEN) )

/** @brief  Enable or disable the APB2 peripheral clock.
  */

#define __RCC_TIM17_CLK_ENABLE()                ( RCC->APB2ENR |= (RCC_APB2ENR_TIM17CKEN) )
#define __RCC_TIM16_CLK_ENABLE()                ( RCC->APB2ENR |= (RCC_APB2ENR_TIM16CKEN) )
#define __RCC_TIM15_CLK_ENABLE()                ( RCC->APB2ENR |= (RCC_APB2ENR_TIM15CKEN) )
#define __RCC_UART1_CLK_ENABLE()                ( RCC->APB2ENR |= (RCC_APB2ENR_UART1CKEN) )
#define __RCC_TIM8_CLK_ENABLE()                 ( RCC->APB2ENR |= (RCC_APB2ENR_TIM8CKEN) )
#define __RCC_TIM1_CLK_ENABLE()                 ( RCC->APB2ENR |= (RCC_APB2ENR_TIM1CKEN) )
#define __RCC_EXTI_CLK_ENABLE()                 ( RCC->APB2ENR |= (RCC_APB2ENR_EXTICKEN) )
#define __RCC_OPA_CLK_ENABLE()                  ( RCC->APB2ENR |= (RCC_APB2ENR_OPACKEN) )
#define __RCC_COMP_CLK_ENABLE()                 do                                              \
                                                {                                               \
                                                    RCC->APB2ENR |= RCC_APB2ENR_COMPCKEN;       \
                                                    RCC->APB2ENR |= (RCC_APB2ENR_EXTICKEN);     \
                                                }while (0)
#define __RCC_VREFBUF_CLK_ENABLE()              ( RCC->APB2ENR |= (RCC_APB2ENR_VREFBUFCKEN) )
#define __RCC_SYSCFG_CLK_ENABLE()               ( RCC->APB2ENR |= (RCC_APB2ENR_SYSCFGCKEN) )

#define __RCC_TIM17_CLK_DISABLE()               ( RCC->APB2ENR &= ~(RCC_APB2ENR_TIM17CKEN) )
#define __RCC_TIM16_CLK_DISABLE()               ( RCC->APB2ENR &= ~(RCC_APB2ENR_TIM16CKEN) )
#define __RCC_TIM15_CLK_DISABLE()               ( RCC->APB2ENR &= ~(RCC_APB2ENR_TIM15CKEN) )
#define __RCC_UART1_CLK_DISABLE()               ( RCC->APB2ENR &= ~(RCC_APB2ENR_UART1CKEN) )
#define __RCC_TIM8_CLK_DISABLE()                ( RCC->APB2ENR &= ~(RCC_APB2ENR_TIM8CKEN) )
#define __RCC_TIM1_CLK_DISABLE()                ( RCC->APB2ENR &= ~(RCC_APB2ENR_TIM1CKEN) )
#define __RCC_EXTI_CLK_DISABLE()                ( RCC->APB2ENR &= ~(RCC_APB2ENR_EXTICKEN) )
#define __RCC_OPA_CLK_DISABLE()                 ( RCC->APB2ENR &= ~(RCC_APB2ENR_OPACKEN) )
#define __RCC_COMP_CLK_DISABLE()                ( RCC->APB2ENR &= ~(RCC_APB2ENR_COMPCKEN) )
#define __RCC_VREFBUF_CLK_DISABLE()             ( RCC->APB2ENR &= ~(RCC_APB2ENR_VREFBUFCKEN) )
#define __RCC_SYSCFG_CLK_DISABLE()              ( RCC->APB2ENR &= ~(RCC_APB2ENR_SYSCFGCKEN) )

/** @brief  Enable or disable the AHB peripheral clock.
  */

#define __RCC_EXMC_CLK_ENABLE()                 ( RCC->AHBENR |= (RCC_AHBENR_EXMCCKEN) )
#define __RCC_ROM_CLK_ENABLE()                  ( RCC->AHBENR |= (RCC_AHBENR_ROMCKEN) )
#define __RCC_FAU_CLK_ENABLE()                  ( RCC->AHBENR |= (RCC_AHBENR_FAUCKEN) )
#define __RCC_HRNG_CLK_ENABLE()                 ( RCC->AHBENR |= (RCC_AHBENR_HRNGCKEN) )
#define __RCC_AES_CLK_ENABLE()                  ( RCC->AHBENR |= (RCC_AHBENR_AESCKEN) )
#define __RCC_DAC1_CLK_ENABLE()                 ( RCC->AHBENR |= (RCC_AHBENR_DAC1CKEN) )
#define __RCC_ADC12_CLK_ENABLE()                ( RCC->AHBENR |= (RCC_AHBENR_ADC12CKEN) )
#define __RCC_GPIOF_CLK_ENABLE()                do                                                      \
                                                {                                                       \
                                                    RCC->AHBENR |= (RCC_AHBENR_GPIOFCKEN);      \
                                                    RCC->APB2ENR |= (RCC_APB2ENR_EXTICKEN);     \
                                                }while (0)
#define __RCC_GPIOE_CLK_ENABLE()                do                                                      \
                                                {                                                       \
                                                    RCC->AHBENR |= (RCC_AHBENR_GPIOECKEN);      \
                                                    RCC->APB2ENR |= (RCC_APB2ENR_EXTICKEN);     \
                                                }while (0)
#define __RCC_GPIOD_CLK_ENABLE()                do                                                      \
                                                {                                                       \
                                                    RCC->AHBENR |= (RCC_AHBENR_GPIODCKEN);      \
                                                    RCC->APB2ENR |= (RCC_APB2ENR_EXTICKEN);     \
                                                }while (0)
#define __RCC_GPIOC_CLK_ENABLE()                do                                                      \
                                                {                                                       \
                                                    RCC->AHBENR |= (RCC_AHBENR_GPIOCCKEN);      \
                                                    RCC->APB2ENR |= (RCC_APB2ENR_EXTICKEN);     \
                                                }while (0)
#define __RCC_GPIOB_CLK_ENABLE()                do                                                      \
                                                {                                                       \
                                                    RCC->AHBENR |= (RCC_AHBENR_GPIOBCKEN);      \
                                                    RCC->APB2ENR |= (RCC_APB2ENR_EXTICKEN);     \
                                                }while (0)
#define __RCC_GPIOA_CLK_ENABLE()                do                                                      \
                                                {                                                       \
                                                    RCC->AHBENR |= (RCC_AHBENR_GPIOACKEN);      \
                                                    RCC->APB2ENR |= (RCC_APB2ENR_EXTICKEN);     \
                                                }while (0)
#define __RCC_SPI3_CLK_ENABLE()                 ( RCC->AHBENR |= (RCC_AHBENR_SPI3CKEN) )
#define __RCC_SPI2_CLK_ENABLE()                 ( RCC->AHBENR |= (RCC_AHBENR_SPI2CKEN) )
#define __RCC_SPI1_CLK_ENABLE()                 ( RCC->AHBENR |= (RCC_AHBENR_SPI1CKEN) )
#define __RCC_CRC_CLK_ENABLE()                  ( RCC->AHBENR |= (RCC_AHBENR_CRCCKEN) )
#define __RCC_USB_CLK_ENABLE()                  do                                              \
                                                {                                               \
                                                    RCC->AHBENR |= (RCC_AHBENR_USBCKEN);        \
                                                    RCC->APB2ENR |= (RCC_APB2ENR_EXTICKEN);     \
                                                }while (0)
#define __RCC_EFC_CLK_ENABLE()                  ( RCC->AHBENR |= (RCC_AHBENR_EFCCKEN) )
#define __RCC_SRAM_CLK_ENABLE()                 ( RCC->AHBENR |= (RCC_AHBENR_SRAMCKEN) )
#define __RCC_DMA2_CLK_ENABLE()                 ( RCC->AHBENR |= (RCC_AHBENR_DMA2CKEN) )
#define __RCC_DMA1_CLK_ENABLE()                 ( RCC->AHBENR |= (RCC_AHBENR_DMA1CKEN) )

#define __RCC_EXMC_CLK_DISABLE()                 ( RCC->AHBENR &= ~(RCC_AHBENR_EXMCCKEN) )
#define __RCC_ROM_CLK_DISABLE()                 ( RCC->AHBENR &= ~(RCC_AHBENR_ROMCKEN) )
#define __RCC_FAU_CLK_DISABLE()                 ( RCC->AHBENR &= ~(RCC_AHBENR_FAUCKEN) )
#define __RCC_HRNG_CLK_DISABLE()                ( RCC->AHBENR &= ~(RCC_AHBENR_HRNGCKEN) )
#define __RCC_AES_CLK_DISABLE()                 ( RCC->AHBENR &= ~(RCC_AHBENR_AESCKEN) )
#define __RCC_DAC1_CLK_DISABLE()                ( RCC->AHBENR &= ~(RCC_AHBENR_DAC1CKEN) )
#define __RCC_ADC12_CLK_DISABLE()               ( RCC->AHBENR &= ~(RCC_AHBENR_ADC12CKEN) )
#define __RCC_GPIOF_CLK_DISABLE()               ( RCC->AHBENR &= ~(RCC_AHBENR_GPIOFCKEN) )
#define __RCC_GPIOE_CLK_DISABLE()               ( RCC->AHBENR &= ~(RCC_AHBENR_GPIOECKEN) )
#define __RCC_GPIOD_CLK_DISABLE()               ( RCC->AHBENR &= ~(RCC_AHBENR_GPIODCKEN) )
#define __RCC_GPIOC_CLK_DISABLE()               ( RCC->AHBENR &= ~(RCC_AHBENR_GPIOCCKEN) )
#define __RCC_GPIOB_CLK_DISABLE()               ( RCC->AHBENR &= ~(RCC_AHBENR_GPIOBCKEN) )
#define __RCC_GPIOA_CLK_DISABLE()               ( RCC->AHBENR &= ~(RCC_AHBENR_GPIOACKEN) )
#define __RCC_SPI3_CLK_DISABLE()                ( RCC->AHBENR &= ~(RCC_AHBENR_SPI3CKEN) )
#define __RCC_SPI2_CLK_DISABLE()                ( RCC->AHBENR &= ~(RCC_AHBENR_SPI2CKEN) )
#define __RCC_SPI1_CLK_DISABLE()                ( RCC->AHBENR &= ~(RCC_AHBENR_SPI1CKEN) )
#define __RCC_CRC_CLK_DISABLE()                 ( RCC->AHBENR &= ~(RCC_AHBENR_CRCCKEN) )
#define __RCC_USB_CLK_DISABLE()                 ( RCC->AHBENR &= ~(RCC_AHBENR_USBCKEN) )
#define __RCC_EFC_CLK_DISABLE()                 ( RCC->AHBENR &= ~(RCC_AHBENR_EFCCKEN) )
#define __RCC_SRAM_CLK_DISABLE()                ( RCC->AHBENR &= ~(RCC_AHBENR_SRAMCKEN) )
#define __RCC_DMA2_CLK_DISABLE()                ( RCC->AHBENR &= ~(RCC_AHBENR_DMA2CKEN) )
#define __RCC_DMA1_CLK_DISABLE()                ( RCC->AHBENR &= ~(RCC_AHBENR_DMA1CKEN) )

/** @brief  Get the enable or disable status of the APB1 peripheral clock.
  */

#define __RCC_LPUART1_IS_CLK_ENABLE()           ( (RCC->APB1ENR & (RCC_APB1ENR_LPUART1CKEN)) != 0U )
#define __RCC_LPTIM1_IS_CLK_ENABLE()            ( (RCC->APB1ENR & (RCC_APB1ENR_LPTIM1CKEN)) != 0U )
#define __RCC_PMU_IS_CLK_ENABLE()               ( (RCC->APB1ENR & (RCC_APB1ENR_PMUCKEN)) != 0U )
#define __RCC_CAN2_IS_CLK_ENABLE()              ( (RCC->APB1ENR & (RCC_APB1ENR_CAN2CKEN)) != 0U )
#define __RCC_CAN1_IS_CLK_ENABLE()              ( (RCC->APB1ENR & (RCC_APB1ENR_CAN1CKEN)) != 0U )
#define __RCC_I2C2_IS_CLK_ENABLE()              ( (RCC->APB1ENR & (RCC_APB1ENR_I2C2CKEN)) != 0U )
#define __RCC_I2C1_IS_CLK_ENABLE()              ( (RCC->APB1ENR & (RCC_APB1ENR_I2C1CKEN)) != 0U )
#define __RCC_UART4_IS_CLK_ENABLE()             ( (RCC->APB1ENR & (RCC_APB1ENR_UART4CKEN)) != 0U )
#define __RCC_UART3_IS_CLK_ENABLE()             ( (RCC->APB1ENR & (RCC_APB1ENR_UART3CKEN)) != 0U )
#define __RCC_UART2_IS_CLK_ENABLE()             ( (RCC->APB1ENR & (RCC_APB1ENR_UART2CKEN)) != 0U )
#define __RCC_I2S2_IS_CLK_ENABLE()              ( (RCC->APB1ENR & (RCC_APB1ENR_I2S2CKEN)) != 0U )
#define __RCC_I2S1_IS_CLK_ENABLE()              ( (RCC->APB1ENR & (RCC_APB1ENR_I2S1CKEN) != 0U )
#define __RCC_WDT_IS_CLK_ENABLE()               ( (RCC->APB1ENR & (RCC_APB1ENR_WDTCKEN)) != 0U )
#define __RCC_RTC_IS_CLK_ENABLE()               ( (RCC->APB1ENR & (RCC_APB1ENR_RTCCKEN)) != 0U )
#define __RCC_TIM7_IS_CLK_ENABLE()              ( (RCC->APB1ENR & (RCC_APB1ENR_TIM7CKEN)) != 0U )
#define __RCC_TIM6_IS_CLK_ENABLE()              ( (RCC->APB1ENR &= (RCC_APB1ENR_TIM6CKEN)) != 0U )
#define __RCC_TIM4_IS_CLK_ENABLE()              ( (RCC->APB1ENR & (RCC_APB1ENR_TIM4CKEN)) != 0U )
#define __RCC_TIM3_IS_CLK_ENABLE()              ( (RCC->APB1ENR & (RCC_APB1ENR_TIM3CKEN)) != 0U )
#define __RCC_TIM2_IS_CLK_ENABLE()              ( (RCC->APB1ENR & (RCC_APB1ENR_TIM2CKEN)) != 0U )

#define __RCC_LPUART1_IS_CLK_DISABLE()          ( (RCC->APB1ENR & (RCC_APB1ENR_LPUART1CKEN)) == 0U )
#define __RCC_LPTIM1_IS_CLK_DISABLE()           ( (RCC->APB1ENR & (RCC_APB1ENR_LPTIM1CKEN)) == 0U )
#define __RCC_PMU_IS_CLK_DISABLE()              ( (RCC->APB1ENR & (RCC_APB1ENR_PMUCKEN)) == 0U )
#define __RCC_CAN2_IS_CLK_DISABLE()             ( (RCC->APB1ENR & (RCC_APB1ENR_CAN2CKEN)) == 0U )
#define __RCC_CAN1_IS_CLK_DISABLE()             ( (RCC->APB1ENR & (RCC_APB1ENR_CAN1CKEN)) == 0U )
#define __RCC_I2C2_IS_CLK_DISABLE()             ( (RCC->APB1ENR & (RCC_APB1ENR_I2C2CKEN)) == 0U )
#define __RCC_I2C1_IS_CLK_DISABLE()             ( (RCC->APB1ENR & (RCC_APB1ENR_I2C1CKEN)) == 0U )
#define __RCC_UART4_IS_CLK_DISABLE()            ( (RCC->APB1ENR & (RCC_APB1ENR_UART4CKEN)) == 0U )
#define __RCC_UART3_IS_CLK_DISABLE()            ( (RCC->APB1ENR & (RCC_APB1ENR_UART3CKEN)) == 0U )
#define __RCC_UART2_IS_CLK_DISABLE()            ( (RCC->APB1ENR & (RCC_APB1ENR_UART2CKEN)) == 0U )
#define __RCC_I2S2_IS_CLK_DISABLE()             ( (RCC->APB1ENR & (RCC_APB1ENR_I2S2CKEN)) == 0U )
#define __RCC_I2S1_IS_CLK_DISABLE()             ( (RCC->APB1ENR & (RCC_APB1ENR_I2S1CKEN) == 0U )
#define __RCC_WDT_IS_CLK_DISABLE()              ( (RCC->APB1ENR & (RCC_APB1ENR_WDTCKEN)) == 0U )
#define __RCC_RTC_IS_CLK_DISABLE()              ( (RCC->APB1ENR & (RCC_APB1ENR_RTCCKEN)) == 0U )
#define __RCC_TIM7_IS_CLK_DISABLE()             ( (RCC->APB1ENR & (RCC_APB1ENR_TIM7CKEN)) == 0U )
#define __RCC_TIM6_IS_CLK_DISABLE()             ( (RCC->APB1ENR &= (RCC_APB1ENR_TIM6CKEN)) == 0U )
#define __RCC_TIM4_IS_CLK_DISABLE()             ( (RCC->APB1ENR & (RCC_APB1ENR_TIM4CKEN)) == 0U )
#define __RCC_TIM3_IS_CLK_DISABLE()             ( (RCC->APB1ENR & (RCC_APB1ENR_TIM3CKEN)) == 0U )
#define __RCC_TIM2_IS_CLK_DISABLE()             ( (RCC->APB1ENR & (RCC_APB1ENR_TIM2CKEN)) == 0U )

/** @brief  Get the enable or disable status of the APB2 peripheral clock.
  */

#define __RCC_TIM17_IS_CLK_ENABLE()             ( (RCC->APB2ENR & (RCC_APB2ENR_TIM17CKEN)) != 0U )
#define __RCC_TIM16_IS_CLK_ENABLE()             ( (RCC->APB2ENR & (RCC_APB2ENR_TIM16CKEN)) != 0U )
#define __RCC_TIM15_IS_CLK_ENABLE()             ( (RCC->APB2ENR & (RCC_APB2ENR_TIM15CKEN)) != 0U )
#define __RCC_UART1_IS_CLK_ENABLE()             ( (RCC->APB2ENR & (RCC_APB2ENR_UART1CKEN)) != 0U )
#define __RCC_TIM8_IS_CLK_ENABLE()              ( (RCC->APB2ENR & (RCC_APB2ENR_TIM8CKEN)) != 0U )
#define __RCC_TIM1_IS_CLK_ENABLE()              ( (RCC->APB2ENR & (RCC_APB2ENR_TIM1CKEN)) != 0U )
#define __RCC_EXTI_IS_CLK_ENABLE()              ( (RCC->APB2ENR & (RCC_APB2ENR_EXTICKEN)) != 0U )
#define __RCC_OPA_IS_CLK_ENABLE()               ( (RCC->APB2ENR & (RCC_APB2ENR_OPACKEN)) != 0U )
#define __RCC_CMP_IS_CLK_ENABLE()               ( (RCC->APB2ENR &= (RCC_APB2ENR_CMPCKEN)) != 0U )
#define __RCC_VREFBUF_IS_CLK_ENABLE()           ( (RCC->APB2ENR & (RCC_APB2ENR_VREFBUFCKEN)) != 0U )
#define __RCC_SYSCFG_IS_CLK_ENABLE()            ( (RCC->APB2ENR & (RCC_APB2ENR_SYSCFGCKEN)) != 0U )

#define __RCC_TIM17_IS_CLK_DISABLE()            ( (RCC->APB2ENR & (RCC_APB2ENR_TIM17CKEN)) == 0U )
#define __RCC_TIM16_IS_CLK_DISABLE()            ( (RCC->APB2ENR & (RCC_APB2ENR_TIM16CKEN)) == 0U )
#define __RCC_TIM15_IS_CLK_DISABLE()            ( (RCC->APB2ENR & (RCC_APB2ENR_TIM15CKEN)) == 0U )
#define __RCC_UART1_IS_CLK_DISABLE()            ( (RCC->APB2ENR & (RCC_APB2ENR_UART1CKEN)) == 0U )
#define __RCC_TIM8_IS_CLK_DISABLE()             ( (RCC->APB2ENR & (RCC_APB2ENR_TIM8CKEN)) == 0U )
#define __RCC_TIM1_IS_CLK_DISABLE()             ( (RCC->APB2ENR & (RCC_APB2ENR_TIM1CKEN)) == 0U )
#define __RCC_EXTI_IS_CLK_DISABLE()             ( (RCC->APB2ENR & (RCC_APB2ENR_EXTICKEN)) == 0U )
#define __RCC_OPA_IS_CLK_DISABLE()              ( (RCC->APB2ENR & (RCC_APB2ENR_OPACKEN)) == 0U )
#define __RCC_CMP_IS_CLK_DISABLE()              ( (RCC->APB2ENR &= (RCC_APB2ENR_CMPCKEN)) == 0U )
#define __RCC_VREFBUF_IS_CLK_DISABLE()          ( (RCC->APB2ENR & (RCC_APB2ENR_VREFBUFCKEN)) == 0U )
#define __RCC_SYSCFG_IS_CLK_DISABLE()           ( (RCC->APB2ENR & (RCC_APB2ENR_SYSCFGCKEN)) == 0U )

/** @brief  Get the enable or disable status of the AHB peripheral clock.
  */

#define __RCC_EXMC_IS_CLK_ENABLE()              ( (RCC->AHBENR & (RCC_AHBENR_EXMCCKEN)) != 0U )
#define __RCC_ROM_IS_CLK_ENABLE()               ( (RCC->AHBENR & (RCC_AHBENR_ROMCKEN)) != 0U )
#define __RCC_FAU_IS_CLK_ENABLE()               ( (RCC->AHBENR & (RCC_AHBENR_FAUCKEN)) != 0U )
#define __RCC_HRNG_IS_CLK_ENABLE()              ( (RCC->AHBENR & (RCC_AHBENR_HRNGCKEN)) != 0U )
#define __RCC_AES_IS_CLK_ENABLE()               ( (RCC->AHBENR & (RCC_AHBENR_AESCKEN)) != 0U )
#define __RCC_DAC1_IS_CLK_ENABLE()              ( (RCC->AHBENR & (RCC_AHBENR_DAC1CKEN)) != 0U )
#define __RCC_ADC12_IS_CLK_ENABLE()             ( (RCC->AHBENR & (RCC_AHBENR_ADC12CKEN)) != 0U )
#define __RCC_GPIOF_IS_CLK_ENABLE()             ( (RCC->AHBENR & (RCC_AHBENR_GPIOFCKEN)) != 0U )
#define __RCC_GPIOE_IS_CLK_ENABLE()             ( (RCC->AHBENR & (RCC_AHBENR_GPIOECKEN)) != 0U )
#define __RCC_GPIOD_IS_CLK_ENABLE()             ( (RCC->AHBENR & (RCC_AHBENR_GPIODCKEN)) != 0U )
#define __RCC_GPIOC_IS_CLK_ENABLE()             ( (RCC->AHBENR & (RCC_AHBENR_GPIOCCKEN)) != 0U )
#define __RCC_GPIOB_IS_CLK_ENABLE()             ( (RCC->AHBENR & (RCC_AHBENR_GPIOBCKEN)) != 0U )
#define __RCC_GPIOA_IS_CLK_ENABLE()             ( (RCC->AHBENR & (RCC_AHBENR_GPIOACKEN)) != 0U )
#define __RCC_SPI3_IS_CLK_ENABLE()              ( (RCC->AHBENR & (RCC_AHBENR_SPI3CKEN)) != 0U )
#define __RCC_SPI2_IS_CLK_ENABLE()              ( (RCC->AHBENR & (RCC_AHBENR_SPI2CKEN)) != 0U )
#define __RCC_SPI1_IS_CLK_ENABLE()              ( (RCC->AHBENR & (RCC_AHBENR_SPI1CKEN)) != 0U )
#define __RCC_CRC_IS_CLK_ENABLE()               ( (RCC->AHBENR & (RCC_AHBENR_CRCCKEN)) != 0U )
#define __RCC_USB_IS_CLK_ENABLE()               ( (RCC->AHBENR & (RCC_AHBENR_USBCKEN)) != 0U )
#define __RCC_EFC_IS_CLK_ENABLE()               ( (RCC->AHBENR & (RCC_AHBENR_EFCCKEN)) != 0U )
#define __RCC_SRAM_IS_CLK_ENABLE()              ( (RCC->AHBENR & (RCC_AHBENR_SRAMCKEN)) != 0U )
#define __RCC_DMA2_IS_CLK_ENABLE()              ( (RCC->AHBENR & (RCC_AHBENR_DMA2CKEN)) != 0U )
#define __RCC_DMA1_IS_CLK_ENABLE()              ( (RCC->AHBENR & (RCC_AHBENR_DMA1CKEN)) != 0U )

#define __RCC_EXMC_IS_CLK_DISABLE()             ( (RCC->AHBENR & (RCC_AHBENR_EXMCCKEN)) == 0U )
#define __RCC_ROM_IS_CLK_DISABLE()              ( (RCC->AHBENR & (RCC_AHBENR_ROMCKEN)) == 0U )
#define __RCC_FAU_IS_CLK_DISABLE()              ( (RCC->AHBENR & (RCC_AHBENR_FAUCKEN)) == 0U )
#define __RCC_HRNG_IS_CLK_DISABLE()             ( (RCC->AHBENR & (RCC_AHBENR_HRNGCKEN)) == 0U )
#define __RCC_AES_IS_CLK_DISABLE()              ( (RCC->AHBENR & (RCC_AHBENR_AESCKEN)) == 0U )
#define __RCC_DAC1_IS_CLK_DISABLE()             ( (RCC->AHBENR & (RCC_AHBENR_DAC1CKEN)) == 0U )
#define __RCC_ADC12_IS_CLK_DISABLE()            ( (RCC->AHBENR & (RCC_AHBENR_ADC12CKEN)) == 0U )
#define __RCC_GPIOF_IS_CLK_DISABLE()            ( (RCC->AHBENR & (RCC_AHBENR_GPIOFCKEN)) == 0U )
#define __RCC_GPIOE_IS_CLK_DISABLE()            ( (RCC->AHBENR & (RCC_AHBENR_GPIOECKEN)) == 0U )
#define __RCC_GPIOD_IS_CLK_DISABLE()            ( (RCC->AHBENR & (RCC_AHBENR_GPIODCKEN)) == 0U )
#define __RCC_GPIOC_IS_CLK_DISABLE()            ( (RCC->AHBENR & (RCC_AHBENR_GPIOCCKEN)) == 0U )
#define __RCC_GPIOB_IS_CLK_DISABLE()            ( (RCC->AHBENR & (RCC_AHBENR_GPIOBCKEN)) == 0U )
#define __RCC_GPIOA_IS_CLK_DISABLE()            ( (RCC->AHBENR & (RCC_AHBENR_GPIOACKEN)) == 0U )
#define __RCC_SPI3_IS_CLK_DISABLE()             ( (RCC->AHBENR & (RCC_AHBENR_SPI3CKEN)) == 0U )
#define __RCC_SPI2_IS_CLK_DISABLE()             ( (RCC->AHBENR & (RCC_AHBENR_SPI2CKEN)) == 0U )
#define __RCC_SPI1_IS_CLK_DISABLE()             ( (RCC->AHBENR & (RCC_AHBENR_SPI1CKEN)) == 0U )
#define __RCC_CRC_IS_CLK_DISABLE()              ( (RCC->AHBENR & (RCC_AHBENR_CRCCKEN)) == 0U )
#define __RCC_USB_IS_CLK_DISABLE()              ( (RCC->AHBENR & (RCC_AHBENR_USBCKEN)) == 0U )
#define __RCC_EFC_IS_CLK_DISABLE()              ( (RCC->AHBENR & (RCC_AHBENR_EFCCKEN)) == 0U )
#define __RCC_SRAM_IS_CLK_DISABLE()             ( (RCC->AHBENR & (RCC_AHBENR_SRAMCKEN)) == 0U )
#define __RCC_DMA2_IS_CLK_DISABLE()             ( (RCC->AHBENR & (RCC_AHBENR_DMA2CKEN)) == 0U )
#define __RCC_DMA1_IS_CLK_DISABLE()             ( (RCC->AHBENR & (RCC_AHBENR_DMA1CKEN)) == 0U )

/******************************************************************************/

/** @defgroup  RCC Private Macros
  * @{
  */

#define IS_RCC_FUNCTIONAL_STATE(_STATE_)        (((_STATE_) == ENABLE)          || \
                                                 ((_STATE_) == DISABLE))

#define IS_RCC_XTH(_XTH_)                       (((_XTH_) == RCC_XTH_OFF)       || \
                                                 ((_XTH_) == RCC_XTH_ON)        || \
                                                 ((_XTH_) == RCC_XTH_BYPASS))

#define IS_RCC_RC64M_DIV(_DIV_)                 (((_DIV_) == RCC_RC64M_DIV1)    || \
                                                 ((_DIV_) == RCC_RC64M_DIV16))

#define IS_RCC_XTL(_XTL_)                       (((_XTL_) == RCC_XTL_OFF)       || \
                                                 ((_XTL_) == RCC_XTL_ON)        || \
                                                 ((_XTL_) == RCC_XTL_BYPASS))

#define IS_RCC_PLLSOURCE(_PLLSOURCE_)           (((_PLLSOURCE_) == RCC_PLLSOURCE_RC64M_DIV16)   || \
                                                 ((_PLLSOURCE_) == RCC_PLLSOURCE_XTH))

#define IS_RCC_PLLM(_PLLM_)                     ((_PLLM_) <= 3U)

#define IS_RCC_PLLN(_PLLN_)                     ((_PLLN_) <= 7U)

#define IS_RCC_PLLF(_PLLF_)                     ((_PLLF_) <= 31U)

#define IS_RCC_SYSCLK_SOURCE(_SOURCE_)          (((_SOURCE_) == RCC_SYSCLKSOURCE_RC64M) || \
                                                 ((_SOURCE_) == RCC_SYSCLKSOURCE_RC32K) || \
                                                 ((_SOURCE_) == RCC_SYSCLKSOURCE_XTH) || \
                                                 ((_SOURCE_) == RCC_SYSCLKSOURCE_XTL) || \
                                                 ((_SOURCE_) == RCC_SYSCLKSOURCE_PLLCLK))

#define IS_RCC_SYSCLK_DIV0(_DIV_)               (((_DIV_) == RCC_SYSDIV0_DIV_1) || \
                                                 ((_DIV_) == RCC_SYSDIV0_DIV_2) || \
                                                 ((_DIV_) == RCC_SYSDIV0_DIV_3) || \
                                                 ((_DIV_) == RCC_SYSDIV0_DIV_4) || \
                                                 ((_DIV_) == RCC_SYSDIV0_DIV_5) || \
                                                 ((_DIV_) == RCC_SYSDIV0_DIV_6) || \
                                                 ((_DIV_) == RCC_SYSDIV0_DIV_7) || \
                                                 ((_DIV_) == RCC_SYSDIV0_DIV_8) || \
                                                 ((_DIV_) == RCC_SYSDIV0_DIV_9) || \
                                                 ((_DIV_) == RCC_SYSDIV0_DIV_10) || \
                                                 ((_DIV_) == RCC_SYSDIV0_DIV_11) || \
                                                 ((_DIV_) == RCC_SYSDIV0_DIV_12) || \
                                                 ((_DIV_) == RCC_SYSDIV0_DIV_13) || \
                                                 ((_DIV_) == RCC_SYSDIV0_DIV_14) || \
                                                 ((_DIV_) == RCC_SYSDIV0_DIV_15) || \
                                                 ((_DIV_) == RCC_SYSDIV0_DIV_16))

#define IS_RCC_SYSCLK_DIV1(_DIV_)               (((_DIV_) == RCC_SYSDIV1_DIV_1) || \
                                                 ((_DIV_) == RCC_SYSDIV1_DIV_2) || \
                                                 ((_DIV_) == RCC_SYSDIV1_DIV_3) || \
                                                 ((_DIV_) == RCC_SYSDIV1_DIV_4) || \
                                                 ((_DIV_) == RCC_SYSDIV1_DIV_5) || \
                                                 ((_DIV_) == RCC_SYSDIV1_DIV_6) || \
                                                 ((_DIV_) == RCC_SYSDIV1_DIV_7) || \
                                                 ((_DIV_) == RCC_SYSDIV1_DIV_8) || \
                                                 ((_DIV_) == RCC_SYSDIV1_DIV_9) || \
                                                 ((_DIV_) == RCC_SYSDIV1_DIV_10) || \
                                                 ((_DIV_) == RCC_SYSDIV1_DIV_11) || \
                                                 ((_DIV_) == RCC_SYSDIV1_DIV_12) || \
                                                 ((_DIV_) == RCC_SYSDIV1_DIV_13) || \
                                                 ((_DIV_) == RCC_SYSDIV1_DIV_14) || \
                                                 ((_DIV_) == RCC_SYSDIV1_DIV_15) || \
                                                 ((_DIV_) == RCC_SYSDIV1_DIV_16))

#define IS_RCC_PCLK1_DIV(_DIV_)                 (((_DIV_) == RCC_PCLK1_DIV_1) || \
                                                 ((_DIV_) == RCC_PCLK1_DIV_2) || \
                                                 ((_DIV_) == RCC_PCLK1_DIV_4) || \
                                                 ((_DIV_) == RCC_PCLK1_DIV_8) || \
                                                 ((_DIV_) == RCC_PCLK1_DIV_16))

#define IS_RCC_PCLK2_DIV(_DIV_)                 (((_DIV_) == RCC_PCLK2_DIV_1) || \
                                                 ((_DIV_) == RCC_PCLK2_DIV_2) || \
                                                 ((_DIV_) == RCC_PCLK2_DIV_4) || \
                                                 ((_DIV_) == RCC_PCLK2_DIV_8) || \
                                                 ((_DIV_) == RCC_PCLK2_DIV_16))

#define IS_RCC_IT(_IT_)                         ((((_IT_) & (RCC_IT_RC32K | RCC_IT_XTL | RCC_IT_RC64M | RCC_IT_XTH | RCC_IT_PLL)) != 0) && \
                                                 (((_IT_) & (~(RCC_IT_RC32K | RCC_IT_XTL | RCC_IT_RC64M | RCC_IT_XTH | RCC_IT_PLL))) == 0))

#define IS_RCC_IT_FLAG(_FLAG_)                  (((_FLAG_) == RCC_IT_FLAG_RC32K) || \
                                                 ((_FLAG_) == RCC_IT_FLAG_XTL) || \
                                                 ((_FLAG_) == RCC_IT_FLAG_RC64M) || \
                                                 ((_FLAG_) == RCC_IT_FLAG_XTH) || \
                                                 ((_FLAG_) == RCC_IT_FLAG_PLL))


#define IS_RCC_MCO_OUTPUT(OUTPUT)               (((OUTPUT) == RCC_MCO_NONE)         || \
                                                 ((OUTPUT) == RCC_MCO_HCLK)         || \
                                                 ((OUTPUT) == RCC_MCO_RC64M)        || \
                                                 ((OUTPUT) == RCC_MCO_RC32K)        || \
                                                 ((OUTPUT) == RCC_MCO_XTH)          || \
                                                 ((OUTPUT) == RCC_MCO_XTL)          || \
                                                 ((OUTPUT) == RCC_MCO_PLLCLK)       || \
                                                 ((OUTPUT) == RCC_MCO_PCLK1)        || \
                                                 ((OUTPUT) == RCC_MCO_PCLK2)        || \
                                                 ((OUTPUT) == RCC_MCO_SYSCLK)       || \
                                                 ((OUTPUT) == RCC_MCO_TIMCLK1)      || \
                                                 ((OUTPUT) == RCC_MCO_TIMCLK2)      || \
                                                 ((OUTPUT) == RCC_MCO_LPUART1CLK)   || \
                                                 ((OUTPUT) == RCC_MCO_FCLKDIV8)     || \
                                                 ((OUTPUT) == RCC_MCO_USBPHYPLL48M) || \
                                                 ((OUTPUT) == RCC_MCO_RTC_PCLK))

#define IS_RCC_MCO1_DIV(_DIV_)                  (((_DIV_) >= 1) && ((_DIV_) <= 65536))

#define IS_RCC_MCO1_POL(_POL_)                  (((_POL_) == RCC_MCO1_POL_LOW)      || \
                                                 ((_POL_) == RCC_MCO1_POL_HIGH))

#define IS_RCC_MCO2(BUZZER2)                    (((BUZZER2) == ENABLE)          || \
                                                 ((BUZZER2) == DISABLE))

#define IS_RCC_MCO2_DIV(_DIV_)                  (((_DIV_) >= 1) && ((_DIV_) <= 64))

#define IS_RCC_MCO2_POL(_POL_)                  (((_POL_) == RCC_MCO2_POL_LOW)  || \
                                                 ((_POL_) == RCC_MCO2_POL_HIGH))

#define RCC_MCO1_POL_LOW                        ( 0 )
#define RCC_MCO1_POL_HIGH                       ( RCC_CLKOCR_BUZZER1EN )

#define RCC_MCO2_POL_LOW                        ( 0 )
#define RCC_MCO2_POL_HIGH                       ( RCC_CLKOCR_BUZZER2EN )
/*
//#define IS_RCC_PLL_PLL(PLL)                     (((PLL) == ENABLE) || \
//                                                 ((PLL) == DISABLE))

//#define IS_RCC_PLL_PLLSOURCE(PLLSOURCE)         (((PLLSOURCE) == RCC_PLLSOURCE_RC64M_DIV16) || \
//                                                 ((PLLSOURCE) == RCC_PLLSOURCE_XTH))


//#define IS_RCC_OSC_OSCTYPE(OSCTYPE)             ((((OSCTYPE) & RCC_OSCTYPE_MASK) != 0U) && (((OSCTYPE) & ~RCC_OSCTYPE_MASK) == 0U))

//#define IS_RCC_OSC_RC64M(RC64M)                 (((RC64M) == ENABLE) || \
//                                                 ((RC64M) == DISABLE))

//#define IS_RCC_OSC_RC64MD_DIV16(DIV16)          (((DIV16) == ENABLE) || \
//                                                 ((DIV16) == DISABLE))

//#define IS_RCC_OSC_RC32K(RC32K)                 (((RC32K) == ENABLE) || \
//                                                 ((RC32K) == DISABLE))

//#define IS_RCC_OSC_XTH(XTH)                     (((XTH) == ENABLE) || \
//                                                 ((XTH) == DISABLE))

//#define IS_RCC_OSC_XTH_BYPASS(BYPASS)           (((BYPASS) == ENABLE) || \
//                                                 ((BYPASS) == DISABLE))

//#define IS_RCC_OSC_XTL(XTL)                     (((XTL) == ENABLE) || \
//                                                 ((XTL) == DISABLE))

//#define IS_RCC_OSC_XTL_BYPASS(BYPASS)           (((BYPASS) == ENABLE) || \
//                                                 ((BYPASS) == DISABLE))

//#define IS_RCC_OSC_XTL_DRIVE(DRIVE)             (((DRIVE) == RCC_XTL_DRIVE0) || \
//                                                 ((DRIVE) == RCC_XTL_DRIVE1) || \
//                                                 ((DRIVE) == RCC_XTL_DRIVE2) || \
//                                                 ((DRIVE) == RCC_XTL_DRIVE3) || \
//                                                 ((DRIVE) == RCC_XTL_LOWPOWER_DRIVE0) || \
//                                                 ((DRIVE) == RCC_XTL_LOWPOWER_DRIVE1) || \
//                                                 ((DRIVE) == RCC_XTL_LOWPOWER_DRIVE2) || \
//                                                 ((DRIVE) == RCC_XTL_LOWPOWER_DRIVE3))

//#define IS_RCC_RTC_CLKSOURCE(SOURCE)            (((SOURCE) == RCC_RTCCLKSOURCE_RC32K) || \
//                                                 ((SOURCE) == RCC_RTCCLKSOURCE_XTL))

//#define IS_RCC_MCO_OUTPUT(OUTPUT)               (((OUTPUT) == RCC_MCO_NONE) || \
//                                                 ((OUTPUT) == RCC_MCO_HCLK) || \
//                                                 ((OUTPUT) == RCC_MCO_RC64M) || \
//                                                 ((OUTPUT) == RCC_MCO_RC32K) || \
//                                                 ((OUTPUT) == RCC_MCO_XTH) || \
//                                                 ((OUTPUT) == RCC_MCO_XTL) || \
//                                                 ((OUTPUT) == RCC_MCO_PLLCLK) || \
//                                                 ((OUTPUT) == RCC_MCO_PCLK1) || \
//                                                 ((OUTPUT) == RCC_MCO_PCLK2) || \
//                                                 ((OUTPUT) == RCC_MCO_SYSCLK) || \
//                                                 ((OUTPUT) == RCC_MCO_TIMCLK1) || \
//                                                 ((OUTPUT) == RCC_MCO_TIMCLK2) || \
//                                                 ((OUTPUT) == RCC_MCO_LPUART1CLK) || \
//                                                 ((OUTPUT) == RCC_MCO_FCLKDIV8) || \
//                                                 ((OUTPUT) == RCC_MCO_USBPHYPLL48M) || \
//                                                 ((OUTPUT) == RCC_MCO_RTC_PCLK))

//#define IS_RCC_MCO1(MCO1)                       (((MCO1) == ENABLE) || \
//                                                 ((MCO1) == DISABLE))

//#define IS_RCC_MCO1_DIV(DIV)                    ((DIV) <= 65535U)

//#define IS_RCC_MCO1_REVPOL(REVPOL)              (((REVPOL) == ENABLE) || \
//                                                 ((REVPOL) == DISABLE))

//#define IS_RCC_MCO2(BUZZER2)                    (((BUZZER2) == ENABLE) || \
//                                                 ((BUZZER2) == DISABLE))

//#define IS_RCC_MCO2_DIV(DIV)                    ((DIV) <= 63U)

//#define IS_RCC_MCO2_REVPOL(REVPOL)              (((REVPOL) == ENABLE) || \
                                                 ((REVPOL) == DISABLE))
*/
#define IS_RCC_LPTIM1_CLKSOURCE(SOURCE)         (((SOURCE) == RCC_LPTIM1CLKSOURCE_PCLK) || \
                                                 ((SOURCE) == RCC_LPTIM1CLKSOURCE_RC32K) || \
                                                 ((SOURCE) == RCC_LPTIM1CLKSOURCE_RC64M) || \
                                                 ((SOURCE) == RCC_LPTIM1CLKSOURCE_XTL))

#define IS_RCC_FLTCLK_CLKSOURCE(SOURCE)         (((SOURCE) == RCC_FLTCLKSOURCE_PCLK_DIV32) || \
                                                 ((SOURCE) == RCC_FLTCLKSOURCE_RC32K))

#define IS_RCC_LPUART1_CLKSOURCE(SOURCE)        (((SOURCE) == RCC_LPUART1CLKSOURCE_RC32K) || \
                                                 ((SOURCE) == RCC_LPUART1CLKSOURCE_XTL) || \
                                                 ((SOURCE) == RCC_LPUART1CLKSOURCE_PCLK_DIV4) || \
                                                 ((SOURCE) == RCC_LPUART1CLKSOURCE_PCLK_DIV8) || \
                                                 ((SOURCE) == RCC_LPUART1CLKSOURCE_PCLK_DIV16) || \
                                                 ((SOURCE) == RCC_LPUART1CLKSOURCE_PCLK_DIV32))
                                                 
                                           
/**
  * @}
  */

/* Exported functions --------------------------------------------------------*/

void RCC_DeInit(void);
void RCC_XTHConfig(uint32_t RCC_XTH);
ErrorStatus RCC_WaitForXTHStartUp(void);
uint32_t RCC_GetXTHFreq(void);
void RCC_RC64MConfig(uint32_t RCC_RC64M_DIV);
void RCC_RC64MCmd(FunctionalState NewState);
ErrorStatus RCC_WaitForRC64MStartUp(void);
uint32_t HAL_RCC_GetRC64MValue(void);
uint32_t RCC_GetRC64MFreq(void);
void RCC_XTLConfig(uint32_t Drive, FunctionalState BypassState);
void RCC_XTLCmd(FunctionalState NewState);
ErrorStatus RCC_WaitForXTLStartUp(void);
uint32_t RCC_GetXTLFreq(void);
void RCC_RC32KCmd(FunctionalState NewState);
ErrorStatus RCC_WaitForRC32KStartUp(void);
uint32_t RCC_GetRC32KFreq(void);
void RCC_PLLConfig(uint32_t RCC_PLLSource, uint32_t PLLM, uint32_t PLLN, uint32_t PLLF);
void RCC_PLLCmd(FunctionalState NewState);
void RCC_PLLUpdateCmd(FunctionalState NewState);
ErrorStatus RCC_WaitForPLLStartUp(void);
uint32_t RCC_GetPLLFreq(void);
void RCC_SYSCLKConfig(uint32_t RCC_SYSCLKSource);
uint32_t RCC_GetSYSCLKSource(void);
void RCC_HCLKConfig(uint32_t RCC_SYSDvi0, uint32_t RCC_SYSDvi1);
void RCC_PCLK1Config(uint32_t RCC_PCLK1Div);
void RCC_PCLK2Config(uint32_t RCC_PCLK2Div);
uint32_t RCC_GetSYSCLKFreq(void);
uint32_t RCC_GetHCLKFreq(void);
uint32_t RCC_GetPCLK1Freq(void);
uint32_t RCC_GetPCLK2Freq(void);
void RCC_GetClocksFreq(RCC_ClocksTypeDef* RCC_Clocks);
void RCC_LPTIM1CLKConfig(uint32_t RCC_LPTIM1CLKSource);
uint32_t RCC_GetLPTIM1CLKFreq(void);
void RCC_FLTCLKConfig(uint32_t ClockSource);
uint32_t RCC_GetFLTCLKFreq(void);
void RCC_LPUART1CLKConfig(uint32_t ClockSource);
uint32_t RCC_GetLPUART1CLKFreq(void);
void RCC_RTCCLKConfig(uint32_t RCC_RTCCLKSource);
void RCC_RTCCLKCmd(FunctionalState NewState);
uint32_t RCC_GetRTCCLKFreq(void);
void RCC_MCO1Config(uint32_t RCC_MCOSource, uint32_t RCC_MCO1Div, uint32_t RCC_MCO1Pol);
void RCC_MCO1Cmd(FunctionalState NewState);
void RCC_MCO2Config(uint32_t RCC_MCOSource, uint32_t RCC_MCO2Div, uint32_t RCC_MCO2Pol);
void RCC_MCO2Cmd(FunctionalState NewState);
void RCC_ITConfig(uint8_t RCC_IT, FunctionalState NewState);
FlagStatus RCC_GetITFlag(uint8_t RCC_IT_Flag);
void RCC_ClearITFlag(uint8_t RCC_IT_Flag);
void RCC_SoftwareReset(void);
void RCC_ClearAllResetFlag(void);
uint32_t RCC_GetResetSource(void);
void RCC_StandbyReset(void);


#ifdef __cplusplus
}
#endif

#endif
