/*
  ******************************************************************************
  * @file    sol_rtc.h
  * @version V1.0.0
  * @date    2023
  * @brief   Header file of RTC SPL module.
  ******************************************************************************
*/

#ifndef __SPL_RTC_H__
#define __SPL_RTC_H__

#include "acm32g103_spl_conf.h"






/**
  * @brief  RTC Configuration Structure definition
  */
typedef struct
{
    uint32_t ClockSource;           /*!< The RTC Clock Source to be configured.*/

    uint32_t Compensation;          /*!< The RTC Clock Compensation to be configured.*/

    uint32_t CompensationValue;     /*!< The RTC Clock Compensation Value to be configured.*/
}RTC_InitTypeDef;


/**
  * @brief  RTC Date structure definition
  */
typedef struct
{
    uint8_t Year;     /*!< Specifies the RTC Date Year. */

    uint8_t Month;    /*!< Specifies the RTC Date Month. */

    uint8_t Date;     /*!< Specifies the RTC Date. */

    uint8_t WeekDay;  /*!< Specifies the RTC Date WeekDay. */
}RTC_DateTypeDef;
/**
  * @}
  */

/**
  * @brief  RTC Time structure definition
  */
typedef struct
{
    uint8_t Hours;          /*!< Specifies the RTC Time Hour. */

    uint8_t Minutes;        /*!< Specifies the RTC Time Minute. */

    uint8_t Seconds;        /*!< Specifies the RTC Time Seconds. */
}RTC_TimeTypeDef;
/**
  * @}
  */

/**
  * @brief  RTC Alarm structure definition
  */
typedef struct
{
    RTC_TimeTypeDef RTC_AlarmTime;     /*!< Specifies the RTC Alarm Time members. */

    uint32_t RTC_AlarmMask;            /*!< Specifies the RTC Alarm Masks. */

    uint32_t RTC_AlarmDateWeekDaySel;  /*!< Specifies the RTC Alarm is on Date or WeekDay. */

    uint8_t RTC_AlarmDateWeekDay;      /*!< Specifies the RTC Alarm Date/WeekDay. */
}RTC_AlarmTypeDef;
/**
  * @}
  */

/**
  * @brief  RTC Temper structure definition
  */
typedef struct 
{
    uint32_t u32_TemperEdge;      /*!< Specifies the RTC Temper edge select. */

    uint32_t u32_InterruptEN;     /*!< Specifies the RTC Temper interrupt enable.*/

    uint32_t u32_ClearBackup;     /*!< Specifies the RTC Temper clear backup register. */
                                       
    uint32_t u32_FilterClk;         /*!< Specifies the RTC Temper Filter Clock select. */

    uint32_t u32_Filter;          /*!< Specifies the RTC Temper Filter select.*/
}RTC_TemperTypeDef;








/** @defgroup RTC Wakeup Timer Definitions 
  * @{
  */ 
#define RTC_WAKEUPCLOCK_RTCCLK_DIV16                (0x00000000U)
#define RTC_WAKEUPCLOCK_RTCCLK_DIV8                 (0x01000000U)
#define RTC_WAKEUPCLOCK_RTCCLK_DIV4                 (0x02000000U)
#define RTC_WAKEUPCLOCK_RTCCLK_DIV2                 (0x03000000U)
#define RTC_WAKEUPCLOCK_1HZ                         (0x04000000U)
#define RTC_WAKEUPCLOCK_0_5HZ                       (0x05000000U)
#define RTC_WAKEUPCLOCK_2HZ                         (0x06000000U)
#define IS_RTC_WAKEUP_CLOCK(CLOCK)                  (((CLOCK) == RTC_WAKEUPCLOCK_RTCCLK_DIV16) || \
                                                    ((CLOCK) == RTC_WAKEUPCLOCK_RTCCLK_DIV8) || \
                                                    ((CLOCK) == RTC_WAKEUPCLOCK_RTCCLK_DIV4) || \
                                                    ((CLOCK) == RTC_WAKEUPCLOCK_RTCCLK_DIV2) || \
                                                    ((CLOCK) == RTC_WAKEUPCLOCK_1HZ) || \
                                                    ((CLOCK) == RTC_WAKEUPCLOCK_0_5HZ) || \
                                                    ((CLOCK) == RTC_WAKEUPCLOCK_2HZ))
#define IS_RTC_WAKEUP_COUNTER(COUNTER)              ((COUNTER) <= 0xFFFF)
/**
  * @}
  */ 
  
/** @defgroup RTC AlarmDateWeekDay Definitions 
  * @{
  */
#define RTC_ALARMDATEWEEKDAYSEL_WEEKDAY             (0x00000000U) 
#define RTC_ALARMDATEWEEKDAYSEL_DATE                (RTC_ALM_ALM_WDS_DAY)
#define IS_RTC_ALARM_DATE_WEEKDAY_SEL(SEL)          (((SEL) == RTC_ALARMDATEWEEKDAYSEL_WEEKDAY) || \
                                                    ((SEL) == RTC_ALARMDATEWEEKDAYSEL_DATE))
/**
  * @}
  */ 


/** @defgroup RTC AlarmMask Definitions 
  * @{
  */ 
#define RTC_ALARMMASK_NONE                          (0x00000000U)
#define RTC_AlLARMMASK_MINUTES                      (0x00000200U)
#define RTC_ALARMMASK_HOUR                          (0x00000400U)
#define RTC_ALARMMASK_DATEWEEKDAY                   (0x00000800U)
#define RTC_ALARMMASK_ALL                           (0x00000E00U)
#define IS_ALARM_MASK(MASK)                         (((MASK) & 0xFFFFF1FF) == (uint32_t)RESET)

/**
  * @}
  */ 



/** @defgroup RTC Time Definitions 
  * @{
  */                                                 
#define IS_RTC_HOUR(HOUR)                           (((HOUR) >= 0) && ((HOUR) <= 0x23))
#define IS_RTC_MINUTES(MINUTES)                     (((MINUTES) >= 0) && ((MINUTES) <= 0x59))
#define IS_RTC_SECONDS(SECONDS)                     (((SECONDS) >= 0) && ((SECONDS) <= 0x59))  
/**
  * @}
  */  

/** @defgroup RTC Date Definitions 
  * @{
  */ 
#define IS_RTC_YEAR(YEAR)                           (((YEAR) >= 0x00) && ((YEAR) <= 0x99))
#define IS_RTC_MONTH(MONTH)                         (((MONTH) >= 0x00) && ((MONTH) <= 0x12))
#define IS_RTC_DAY(DAY)                             (((DAY) >= 0x00) && ((DAY) <= 0x31))
#define IS_RTC_WEEKDAY(WEEKDAY)                     (((WEEKDAY) >= 0x00) && ((WEEKDAY) <= 0x07))
/**
  * @}
  */                                        

/** @defgroup RTC Input parameter format_definitions 
  * @{
  */ 
#define RTC_FORMAT_DECIMAL                          (0U)
#define RTC_FORMAT_BCD                              (1U)
#define IS_RTC_FORMAT(FORMAT)                       (((FORMAT) == RTC_FORMAT_DECIMAL) || ((FORMAT) == RTC_FORMAT_BCD))

/**
  * @}
  */ 

/** @defgroup ClockSource
  * @{
  */
#define RTC_CLOCK_RC32K                             (0U)
#define RTC_CLOCK_XTL                               (1U)  
#define IS_RTC_CLOCKSRC(__CLOCKSRC__)               (((__CLOCKSRC__) == RTC_CLOCK_RC32K) || ((__CLOCKSRC__) == RTC_CLOCK_XTL))
/**
  * @}
  */                                            

/** @defgroup RTC Interrupts Definitions 
  * @{
  */ 
#define RTC_IT_FLAG_DATE                            (0x00000001UL)  
#define RTC_IT_FLAG_HOUR                            (0x00000002UL)  
#define RTC_IT_FLAG_MIN                             (0x00000004UL)  
#define RTC_IT_FLAG_SEC                             (0x00000008UL) 
#define RTC_IT_FLAG_2HZ                             (0x00000010UL) 
#define RTC_IT_FLAG_4HZ                             (0x00000020UL) 
#define RTC_IT_FLAG_8HZ                             (0x00000040UL)    
#define RTC_IT_FLAG_16HZ                            (0x00000080UL)  
#define RTC_IT_FLAG_64HZ                            (0x00000100UL)  
#define RTC_IT_FLAG_256HZ                           (0x00000200UL)  
#define RTC_IT_FLAG_1KHZ                            (0x00000400UL)  
#define RTC_IT_FLAG_ALM                             (0x00000800UL)  
#define RTC_IT_FLAG_32S                             (0x00001000UL)  
#define RTC_IT_FLAG_STP1F                           (0x00002000UL)  
#define RTC_IT_FLAG_STP1R                           (0x00004000UL)  
#define RTC_IT_FLAG_STP2F                           (0x00008000UL)  
#define RTC_IT_FLAG_STP2R                           (0x00010000UL)  
#define RTC_IT_FLAG_WUT                             (0x00020000UL)
#define IS_RTC_IT_FLAG(FLAG)                        (((FLAG) == RTC_IT_FLAG_DATE) || ((FLAG) == RTC_IT_FLAG_HOUR) || \
                                                    ((FLAG) == RTC_IT_FLAG_MIN)|| ((FLAG) == RTC_IT_FLAG_SEC) || \
                                                    ((FLAG) == RTC_IT_FLAG_2HZ)|| ((FLAG) == RTC_IT_FLAG_4HZ) || \
                                                    ((FLAG) == RTC_IT_FLAG_8HZ)|| ((FLAG) == RTC_IT_FLAG_16HZ) || \
                                                    ((FLAG) == RTC_IT_FLAG_64HZ)|| ((FLAG) == RTC_IT_FLAG_256HZ) || \
                                                    ((FLAG) == RTC_IT_FLAG_1KHZ)|| ((FLAG) == RTC_IT_FLAG_ALM) || \
                                                    ((FLAG) == RTC_IT_FLAG_32S)|| ((FLAG) == RTC_IT_FLAG_STP1F) || \
                                                    ((FLAG) == RTC_IT_FLAG_STP1R)|| ((FLAG) == RTC_IT_FLAG_STP2F) || \
                                                    ((FLAG) == RTC_IT_FLAG_STP2R)|| ((FLAG) == RTC_IT_FLAG_WUT))   
/**
  * @}
  */ 
  

/** @defgroup RTC Output selection Definitions 
  * @{
  */ 
#define RTC_OUTPUT_NONE                             (0U)  
#define RTC_OUTPUT_SECOND                           (2U)
#define RTC_OUTPUT_MINUTE                           (3U)
#define RTC_OUTPUT_HOUR                             (4U)
#define RTC_OUTPUT_DAY                              (5U)
#define RTC_OUTPUT_ALARM                            (6U)
#define RTC_OUTPUT_32_SECONDS                       (7U)
#define RTC_OUTPUT_WAKEUP                           (8U)
#define IS_RTC_OUTPUT(OUTPUT)                       (((OUTPUT) == RTC_OUTPUT_NONE) || \
                                                    ((OUTPUT) == RTC_OUTPUT_SECOND) || \
                                                    ((OUTPUT) == RTC_OUTPUT_MINUTE) || \
                                                    ((OUTPUT) == RTC_OUTPUT_HOUR) || \
                                                    ((OUTPUT) == RTC_OUTPUT_DAY) || \
                                                    ((OUTPUT) == RTC_OUTPUT_ALARM) || \
                                                    ((OUTPUT) == RTC_OUTPUT_32_SECONDS) || \
                                                    ((OUTPUT) == RTC_OUTPUT_WAKEUP))

/**
  * @}
  */

/** @defgroup RTC Digital Calibration Definitions 
  * @{
  */ 
#define RTC_CALIBSIGN_POSITIVE                      (0x00000000U) 
#define RTC_CALIBSIGN_NEGATIVE                      (RTC_ADJUST_ADJSIGN_DEC)
#define IS_RTC_CALIB_SIGN(SIGN)                     (((SIGN) == RTC_CALIBSIGN_POSITIVE) || ((SIGN) == RTC_CALIBSIGN_NEGATIVE))
#define IS_RTC_CALIB_VALUE(VALUE)                   ((VALUE) < 0x1FF)
/**
  * @}
  */   

/** @defgroup RTC Tamper Trigger Definitions 
  * @{
  */ 
#define RTC_TAMPERTRIGGER_RISING                    (0U)
#define RTC_AMPERTRIGGER_FALLING                    (1U)
#define IS_RTC_TAMPER_TRIGGER(TRIGGER)              (((TRIGGER) == RTC_TAMPERTRIGGER_RISING) || \
                                                    ((TRIGGER) == RTC_AMPERTRIGGER_FALLING))       
/**
  * @}
  */ 

/** @defgroup RTC Tamper Filter Definitions 
  * @{
  */ 
#define RTC_TAMPERFILTER_1SAMPLE                    (0U) 
#define RTC_TAMPERFILTER_2SAMPLE                    (1U)
#define RTC_TAMPERFILTER_4SAMPLE                    (2U)                                    
#define RTC_TAMPERFILTER_8SAMPLE                    (3U) 
#define IS_RTC_TAMPER_FILTER(FILTER)                (((FILTER) == RTC_TAMPERFILTER_1SAMPLE) || \
                                                    ((FILTER) == RTC_TAMPERFILTER_2SAMPLE) || \
                                                    ((FILTER) == RTC_TAMPERFILTER_4SAMPLE) || \
                                                    ((FILTER) == RTC_TAMPERFILTER_8SAMPLE))
/**
  * @}
  */ 
  
  /** @defgroup RTC Tamper Filter status 
  * @{
  */ 
#define RTC_TAMPERFILTER_DISABLE                    (0U) 
#define RTC_TAMPERFILTER_ENABLE                     (1U)
#define IS_RTC_TAMPER_FILTER_STATUS(STATUS)         (((STATUS) == RTC_TAMPERFILTER_DISABLE) || \
                                                    ((STATUS) == RTC_TAMPERFILTER_ENABLE))
/**
  * @}
  */ 

/** @defgroup RTC Tamper Sampling clock Definitions 
  * @{
  */ 
#define RTC_TAMPERSAMPLINGCLOCK_RTCCLK              (0x00000000U)                                                                   
#define RTC_TamperSamplingFreq_512_RTCCLK           (RTC_CR_TAMPFLTCLK_512_RTCCLK) 
#define IS_RTC_TAMPER_FILTER_CLOCK(CLOCK)           (((CLOCK) == RTC_TAMPERSAMPLINGCLOCK_RTCCLK) || \
                                                    ((CLOCK) == RTC_TamperSamplingFreq_512_RTCCLK))
/**
  * @}
  */

/** @defgroup RTC Tamper Definitions 
  * @{
  */ 
#define RTC_TAMPER_1                                (RTC_CR_TAMP1EN)
#define RTC_TAMPER_2                                (RTC_CR_TAMP2EN)
#define IS_RTC_TAMPER(TAMPER)                       (((TAMPER) == RTC_TAMPER_1) || ((TAMPER) == RTC_TAMPER_2))
/**
  * @}
  */
  
/** @defgroup RTC Tamper Falling edge clear backup Definitions
  * @{
  */ 
#define RTC_TAMPERFALLINGCLEAR_DISABLE              (0U)
#define RTC_TAMPERFALLINGCLEAR_ENABLE               (1U)
#define IS_RTC_TAMPER_FALLING_CLEAR(FALL)           (((FALL) == RTC_TAMPERFALLINGCLEAR_ENABLE) || \
                                                    ((FALL) == RTC_TAMPERFALLINGCLEAR_DISABLE))
/**
  * @}
  */
  
/** @defgroup RTC Tamper Rasing edge clear backup Definitions
* @{
*/ 
#define RTC_TAMPERRASINGCLEAR_DISABLE               (0U)
#define RTC_TAMPERRASINGCLEAR_ENABLE                (1U)
#define IS_RTC_TAMPER_RASING_CLEAR(RASE)            (((RASE) == RTC_TAMPERRASINGCLEAR_DISABLE) || \
                                                    ((RASE) == RTC_TAMPERRASINGCLEAR_ENABLE))
/**
  * @}
  */
  
/** @defgroup RTC_Backup_Registers_Definitions 
* @{
*/

#define RTC_BKP_DR0                       ((uint32_t)0x00000000)
#define RTC_BKP_DR1                       ((uint32_t)0x00000001)
#define RTC_BKP_DR2                       ((uint32_t)0x00000002)
#define RTC_BKP_DR3                       ((uint32_t)0x00000003)
#define RTC_BKP_DR4                       ((uint32_t)0x00000004)
#define RTC_BKP_DR5                       ((uint32_t)0x00000005)
#define RTC_BKP_DR6                       ((uint32_t)0x00000006)
#define RTC_BKP_DR7                       ((uint32_t)0x00000007)
#define RTC_BKP_DR8                       ((uint32_t)0x00000008)
#define RTC_BKP_DR9                       ((uint32_t)0x00000009)
#define RTC_BKP_DR10                      ((uint32_t)0x0000000A)
#define RTC_BKP_DR11                      ((uint32_t)0x0000000B)
#define RTC_BKP_DR12                      ((uint32_t)0x0000000C)
#define RTC_BKP_DR13                      ((uint32_t)0x0000000D)
#define RTC_BKP_DR14                      ((uint32_t)0x0000000E)
#define RTC_BKP_DR15                      ((uint32_t)0x0000000F)
#define IS_RTC_BKP(BKP)                   (((BKP) == RTC_BKP_DR0) || \
                                           ((BKP) == RTC_BKP_DR1) || \
                                           ((BKP) == RTC_BKP_DR2) || \
                                           ((BKP) == RTC_BKP_DR3) || \
                                           ((BKP) == RTC_BKP_DR4) || \
                                           ((BKP) == RTC_BKP_DR5) || \
                                           ((BKP) == RTC_BKP_DR6) || \
                                           ((BKP) == RTC_BKP_DR7) || \
                                           ((BKP) == RTC_BKP_DR8) || \
                                           ((BKP) == RTC_BKP_DR9) || \
                                           ((BKP) == RTC_BKP_DR10) || \
                                           ((BKP) == RTC_BKP_DR11) || \
                                           ((BKP) == RTC_BKP_DR12) || \
                                           ((BKP) == RTC_BKP_DR13) || \
                                           ((BKP) == RTC_BKP_DR14) || \
                                           ((BKP) == RTC_BKP_DR15))
/**
  * @}
  */ 
  
  
  
  
/*  Function used to set the RTC configuration to the default reset state */
void RTC_DeInit(void);

/* Initialization and Configuration functions */
void RTC_Init(uint32_t ClockSource);
void RTC_WriteProtectionCmd(FunctionalState NewState);

/* Time and Date configuration functions */
void RTC_SetTime(uint32_t RTC_Format, RTC_TimeTypeDef* RTC_TimeStruct);
void RTC_TimeStructInit(RTC_TimeTypeDef* RTC_TimeStruct);
void RTC_GetTime(uint32_t RTC_Format, RTC_TimeTypeDef* RTC_TimeStruct);
void RTC_SetDate(uint32_t RTC_Format, RTC_DateTypeDef* RTC_DateStruct);
void RTC_DateStructInit(RTC_DateTypeDef* RTC_DateStruct);
void RTC_GetDate(uint32_t RTC_Format, RTC_DateTypeDef* RTC_DateStruct);

/* Alarm configuration functions */
void RTC_SetAlarm(uint32_t RTC_Format, RTC_AlarmTypeDef* RTC_AlarmStruct);
void RTC_AlarmStructInit(RTC_AlarmTypeDef* RTC_AlarmStruct);
void RTC_GetAlarm(uint32_t RTC_Format, RTC_AlarmTypeDef* RTC_AlarmStruct);
void RTC_AlarmCmd(FunctionalState NewState);

/* Wakeup Timer configuration functions */
void RTC_WakeUpClockConfig(uint32_t RTC_WakeUpClock);
void RTC_SetWakeUpCounter(uint32_t RTC_WakeUpCounter);
uint32_t RTC_GetWakeUpCounter(void);
void RTC_WakeUpCmd(FunctionalState NewState);

/* Output pin Configuration function */
void RTC_OutputConfig(uint32_t RTC_Output);

/* Digital Calibration configuration functions */
void RTC_CoarseCalibConfig(uint32_t RTC_CalibSign, uint32_t Value);

/* Tamper configuration functions */
void RTC_TamperTriggerConfig(uint32_t RTC_Tamper, uint32_t RTC_TamperTrigger);
void RTC_TamperCmd(uint32_t RTC_Tamper, FunctionalState NewState);
void RTC_TamperFilterPeriodConfig(uint32_t RTC_Tamper, uint32_t RTC_TamperFilter);
void RTC_TamperFilterClockConfig(uint32_t RTC_TamperSamplingClock);
void RTC_TamperFallingClearBackpConfig(uint32_t RTC_Tamper, uint32_t RTC_TamperFallingClear);
void RTC_TamperRasingClearBackpConfig(uint32_t RTC_Tamper, uint32_t RTC_TamperRasingClear);

/* Backup Data Registers configuration functions */
void RTC_WriteBackupRegister(uint32_t RTC_BKP_DR, uint32_t Data);
uint32_t RTC_ReadBackupRegister(uint32_t RTC_BKP_DR);

/* Interrupts and flags management functions */
void RTC_ITConfig(uint32_t RTC_IT, FunctionalState NewState);
FlagStatus RTC_GetFlagStatus(uint32_t RTC_FLAG);
void RTC_ClearFlag(uint32_t RTC_FLAG);
ITStatus RTC_GetITStatus(uint32_t RTC_IT);
void RTC_ClearITPendingBit(uint32_t RTC_IT);












//#endif    /* SPL_RTC_MODULE_ENABLED */


/************************ (C) COPYRIGHT AISINOCHIP *****END OF FILE****/




  
  
  
  
  


#endif /* __SPL_RTC_H__ */


/************************ (C) COPYRIGHT AISINOCHIP *****END OF FILE****/


