
/******************************************************************************
*@file  : spl_dma.c
*@brief : DMA SPL module driver.
******************************************************************************/

#include "acm32g103_spl_conf.h" 

/******************************************************************************
*@brief : DeInitializes the DMA peripheral.
*@param : DMAy_Channelx: where y can be 1 or 2 to select the DMA and 
*                              x can be 0 to 7 to select the DMA Channel.
*@return: None
******************************************************************************/
void DMA_DeInit(DMA_Channel_TypeDef *DMAy_Channelx)
{
    uint32_t Channel;
    DMA_TypeDef *DMAy;
    
    /* Check the parameters */
	
	DMAy_Channelx->CHCONFIG   = 0U;
	DMAy_Channelx->CHCTRL     = 0U;
	DMAy_Channelx->CHSRCADDR  = 0U;
	DMAy_Channelx->CHDESTADDR = 0U;
	DMAy_Channelx->CHLLI      = 0U;
	
    /* calculation of the channel */
	if ((uint32_t)(DMAy_Channelx) < (uint32_t)(DMA2_Channel0))
	{
		Channel = 1UL << (((uint32_t)(DMAy_Channelx) - (uint32_t)(DMA1_Channel0)) >> 5);
		DMAy = DMA1;
	}
	else
	{
		Channel = 1UL << (((uint32_t)(DMAy_Channelx) - (uint32_t)(DMA2_Channel0)) >> 5);
		DMAy = DMA2;
	}
    
    DMAy->INTTCCLR = (Channel << 8U) + Channel;
    DMAy->INTERRCLR = Channel;
}

/******************************************************************************
*@brief : Initializes the DMAy Channelx according to the specified parameters in 
*         the DMA_InitStruct structure.
*@param : DMAy_Channelx: where y can be 1 or 2 to select the DMA and 
*                              x can be 0 to 7 to select the DMA Channel.
*@param : DMA_InitStruct: pointer to a DMA_InitTypeDef structure that contains
*         the configuration information for the specified DMA Channel.
*@return: None
******************************************************************************/
void DMA_Init(DMA_Channel_TypeDef *DMAy_Channelx, DMA_InitTypeDef *DMA_InitStruct) 
{
    uint32_t temp;
//    uint32_t Channel;
	uint32_t RequestID;
    DMA_TypeDef *DMAy;
    
    /* Check the parameters */

    /* calculation of the channel */
	if ((uint32_t)(DMAy_Channelx) < (uint32_t)(DMA2_Channel0))
	{
//		Channel = 1UL << (((uint32_t)(DMAy_Channelx) - (uint32_t)(DMA1_Channel0)) >> 5);
		DMAy = DMA1;
	}
	else
	{
//		Channel = 1UL << (((uint32_t)(DMAy_Channelx) - (uint32_t)(DMA2_Channel0)) >> 5);
		DMAy = DMA2;
	}
    DMAy->CONFIG |= DMA_CHCONFIG_EN;
    
    /* Disable Channel */
    DMAy_Channelx->CHCONFIG &= ~DMA_CHCONFIG_EN; 
    
    if (DMA_InitStruct->DataFlow == DMA_DATAFLOW_M2P) 
    {
        RequestID = DMA_InitStruct->ReqID << DMA_CHCONFIG_DESTPERIPH_Pos;
    }
    else if (DMA_InitStruct->DataFlow == DMA_DATAFLOW_P2M) 
    {
        RequestID = DMA_InitStruct->ReqID << DMA_CHCONFIG_SRCPERIPH_Pos;
    }
	else
	{
		RequestID = 0;
	}
    
    temp = DMAy_Channelx->CHCONFIG;
    temp &= ~(DMA_CHCONFIG_SRCPERIPH | DMA_CHCONFIG_DESTPERIPH | DMA_CHCONFIG_FLOWCTRL);
    temp |= DMA_InitStruct->DataFlow | RequestID;
    DMAy_Channelx->CHCONFIG = temp;
    
    /* Config Channel Control */
    DMAy_Channelx->CHCTRL = DMA_InitStruct->RawInt | DMA_InitStruct->Size | \
                            DMA_InitStruct->DestInc | DMA_InitStruct->SrcInc | \
                            DMA_InitStruct->DestWidth | DMA_InitStruct->SrcWidth | \
                            DMA_InitStruct->SrcBurst | DMA_InitStruct->DestBurst;
    
    DMAy_Channelx->CHSRCADDR = DMA_InitStruct->SrcAddr;
    DMAy_Channelx->CHDESTADDR = DMA_InitStruct->DestAddr;
    
    if (DMA_InitStruct->Mode == DMA_MODE_NORMAL)
    {
        DMAy_Channelx->CHLLI = 0;
    }
    else
    {
        DMA_InitStruct->Link.SrcAddr = DMAy_Channelx->CHSRCADDR;
        DMA_InitStruct->Link.DestAddr = DMAy_Channelx->CHDESTADDR;
        DMA_InitStruct->Link.Ctrl = (DMAy_Channelx->CHCTRL & ~DMA_CHCTRL_TRANSFERSIZE) | DMA_InitStruct->Size;
        DMA_InitStruct->Link.Next = &DMA_InitStruct->Link;
        
        DMAy_Channelx->CHLLI = (uint32_t)&DMA_InitStruct->Link;
    }
}

/******************************************************************************
*@brief : Fills each DMA_InitStruct member with its default value.
*@param : DMA_InitStruct: pointer to a DMA_InitTypeDef structure that contains
*         the configuration information for the specified DMA Channel.
*@return: None
******************************************************************************/
void DMA_StructInit(DMA_InitTypeDef* DMA_InitStruct)
{
    DMA_InitStruct->Mode        = DMA_MODE_NORMAL;
    DMA_InitStruct->DataFlow    = DMA_DATAFLOW_M2M;
    DMA_InitStruct->ReqID       = 0;
    DMA_InitStruct->SrcAddr     = 0;
    DMA_InitStruct->DestAddr    = 0;
    DMA_InitStruct->Size        = 0;
    DMA_InitStruct->RawInt      = DMA_RAWINT_ENABLE;
    DMA_InitStruct->SrcWidth    = DMA_SRCWIDTH_BYTE;
    DMA_InitStruct->DestWidth   = DMA_DESTWIDTH_BYTE;
    DMA_InitStruct->SrcInc      = DMA_SRCINC_DISABLE;
    DMA_InitStruct->DestInc     = DMA_DESTINC_DISABLE;
    DMA_InitStruct->SrcBurst    = DMA_SRCBURST_1;
    DMA_InitStruct->DestBurst   = DMA_DESTBURST_1;
}


/******************************************************************************
*@brief : Enables or disables the big endian mode.
*@param : DMAy: where y can be 1 or 2 to select the DMA.
*@param : NewState: new state of the big endian mode. 
*         This parameter can be: ENABLE or DISABLE.
*@return: None
******************************************************************************/
void DMA_BigEndianConfig(DMA_TypeDef *DMAy, FunctionalState NewState)
{
    if (NewState != DISABLE)
        DMAy->CONFIG |= (uint32_t)DMA_CONFIG_ENDIAN;
    else
        DMAy->CONFIG &= ~(uint32_t)DMA_CONFIG_ENDIAN;
}

/******************************************************************************
*@brief : Configures the source address.
*@param : DMAy_Channelx: where y can be 1 or 2 to select the DMA and 
*                              x can be 0 to 7 to select the DMA Channel.
*@param : SrcAddr: the source address. 
*@return: None
******************************************************************************/
void DMA_SrcAddrConfig(DMA_Channel_TypeDef *DMAy_Channelx, uint32_t SrcAddr)
{
    DMAy_Channelx->CHSRCADDR = SrcAddr;
}

/******************************************************************************
*@brief : Configures the desination address.
*@param : DMAy_Channelx: where y can be 1 or 2 to select the DMA and 
*                              x can be 0 to 7 to select the DMA Channel.
*@param : DestAddr: the desination address. 
*@return: None
******************************************************************************/
void DMA_DestAddrConfig(DMA_Channel_TypeDef *DMAy_Channelx, uint32_t DestAddr)
{
    DMAy_Channelx->CHDESTADDR = DestAddr;
}

/******************************************************************************
*@brief : Writes the number of data units to be transferred on the DMAy Channelx.
*@param : DMAy_Channelx: where y can be 1 or 2 to select the DMA and 
*                              x can be 0 to 7 to select the DMA Channel.
*@param : Size: Number of data units to be transferred (from 1 to 65535)
*@return: None
******************************************************************************/
void DMA_SetTransferSize(DMA_Channel_TypeDef* DMAy_Channelx, uint16_t Size)
{
    DMAy_Channelx->CHCTRL = (DMAy_Channelx->CHCTRL & ~(uint32_t)DMA_CHCTRL_TRANSFERSIZE) | Size;
}

/******************************************************************************
*@brief : Returns the number of remaining data units in the current DMAy Channelx transfer.
*@param : DMAy_Channelx: where y can be 1 or 2 to select the DMA and 
*                              x can be 0 to 7 to select the DMA Channel.
*@return: The number of remaining data units in the current DMAy Channelx transfer.
******************************************************************************/
uint16_t DMA_GetTransferSize(DMA_Channel_TypeDef* DMAy_Channelx)
{
    return ((uint16_t)(DMAy_Channelx->CHCTRL & DMA_CHCTRL_TRANSFERSIZE));
}

/******************************************************************************
*@brief : Enables or disables the specified DMAy Channelx.
*@param : DMAy_Channelx: where y can be 1 or 2 to select the DMA and 
*                              x can be 0 to 7 to select the DMA Channel.
*@param : NewState: new state of the specified DMAy. 
*         This parameter can be: ENABLE or DISABLE.
*@return: None
******************************************************************************/
void DMA_Cmd(DMA_Channel_TypeDef *DMAy_Channelx, FunctionalState NewState)
{
    if (NewState != DISABLE)
        DMAy_Channelx->CHCONFIG |= (uint32_t)DMA_CHCONFIG_EN;
    else
        DMAy_Channelx->CHCONFIG &= ~(uint32_t)DMA_CHCONFIG_EN;
}



/******************************************************************************
*@brief : Configures linked list.
*@param : Link: Initialize linked list node.
*@param : Link_Init: pointer to an DMA_LinkInitTypeDef structure that contains 
*                    the configuration information for the specified Link.
*@return: None
******************************************************************************/
void DMA_LinkConfig(DMA_LinkTypeDef *Link, DMA_LinkInitTypeDef *Link_Init)
{
    /* Check the parameters */
	assert_param(Link != NULL);
	assert_param(IS_DMA_SRCADDR(Link_Init->SrcAddr));
	assert_param(IS_DMA_DESTADDR(Link_Init->DestAddr));
	assert_param(IS_DMA_RAWIT(Link_Init->RawInt));
	assert_param(IS_DMA_SRCINC(Link_Init->SrcInc));
	assert_param(IS_DMA_DESTINC(Link_Init->DestInc));
	assert_param(IS_DMA_SRCWIDTH(Link_Init->SrcWidth));
	assert_param(IS_DMA_DESTWIDTH(Link_Init->DestWidth));
	assert_param(IS_DMA_SRCBURST(Link_Init->SrcBurst));
	assert_param(IS_DMA_DESTBURST(Link_Init->DestBurst));
	assert_param(IS_DMA_SIZE(Link_Init->Size));

	Link->SrcAddr = Link_Init->SrcAddr;
	Link->DestAddr = Link_Init->DestAddr;
	Link->Next = Link_Init->Next;
	Link->Ctrl = Link_Init->RawInt | Link_Init->SrcInc | Link_Init->DestInc | \
                 Link_Init->SrcWidth | Link_Init->DestWidth | Link_Init->SrcBurst | \
                 Link_Init->DestBurst | Link_Init->Size ;
}

/******************************************************************************
*@brief : Initializes the DMAy Channelx according to the specified parameters in 
*         the DMA_InitStruct structure.
*@param : DMAy_Channelx: where y can be 1 or 2 to select the DMA and 
*                              x can be 0 to 7 to select the DMA Channel.
*@param : DMA_InitStruct: pointer to the DMA_InitTypeDef structure, which contains 
*                         the configuration information of the specified DMA channel, 
*                         but only including the flow controller, the periph id.
*@param : Link: pointer to the DMA_LinkTypeDef structure, which contains the configuration
*                         information of the specified DMA channel.
*@return: None
******************************************************************************/
void DMA_LinkInit(DMA_Channel_TypeDef *DMAy_Channelx, DMA_InitTypeDef *DMA_InitStruct, \
                                                      DMA_LinkTypeDef *Link) 
{
    uint32_t temp;
//    uint32_t Channel;
	uint32_t RequestID;
    DMA_TypeDef *DMAy;
    
    /* Check the parameters */

    /* calculation of the channel */
	if ((uint32_t)(DMAy_Channelx) < (uint32_t)(DMA2_Channel0))
	{
//		Channel = 1UL << (((uint32_t)(DMAy_Channelx) - (uint32_t)(DMA1_Channel0)) >> 5);
		DMAy = DMA1;
	}
	else
	{
//		Channel = 1UL << (((uint32_t)(DMAy_Channelx) - (uint32_t)(DMA2_Channel0)) >> 5);
		DMAy = DMA2;
	}
    DMAy->CONFIG |= DMA_CHCONFIG_EN;
    
    /* Disable Channel */
    DMAy_Channelx->CHCONFIG &= ~DMA_CHCONFIG_EN; 
    
    if (DMA_InitStruct->DataFlow == DMA_DATAFLOW_M2P) 
    {
        RequestID = DMA_InitStruct->ReqID << DMA_CHCONFIG_DESTPERIPH_Pos;
    }
    else if (DMA_InitStruct->DataFlow == DMA_DATAFLOW_P2M) 
    {
        RequestID = DMA_InitStruct->ReqID << DMA_CHCONFIG_SRCPERIPH_Pos;
    }
	else
	{
		RequestID = 0;
	}
    
    temp = DMAy_Channelx->CHCONFIG;
    temp &= ~(DMA_CHCONFIG_SRCPERIPH | DMA_CHCONFIG_DESTPERIPH | DMA_CHCONFIG_FLOWCTRL);
    temp |= DMA_InitStruct->DataFlow | RequestID;
    DMAy_Channelx->CHCONFIG = temp;
    
    DMAy_Channelx->CHSRCADDR = Link->SrcAddr;
    DMAy_Channelx->CHDESTADDR = Link->DestAddr;
    DMAy_Channelx->CHLLI = (uint32_t)Link->Next;
    DMAy_Channelx->CHCTRL = Link->Ctrl;
}

/******************************************************************************
*@brief : Enables or disables the specified DMAy Channelx transfer abort command.
*@param : DMAy_Channelx: where y can be 1 or 2 to select the DMA and 
*                              x can be 0 to 7 to select the DMA Channel.
*@param : NewState: new state of the transfer abort command. 
*         This parameter can be: ENABLE or DISABLE.
*@return: None
******************************************************************************/
void DMA_AbortCmd(DMA_Channel_TypeDef *DMAy_Channelx, FunctionalState NewState)
{
    if (NewState != DISABLE)
        DMAy_Channelx->CHCONFIG |= (uint32_t)DMA_CHCONFIG_HALT;
    else
        DMAy_Channelx->CHCONFIG &= ~(uint32_t)DMA_CHCONFIG_HALT;
}

/******************************************************************************
*@brief : Returns the enable status of the specified DMAy Channelx.
*@param : DMAy_Channelx: where y can be 1 or 2 to select the DMA and 
*                              x can be 0 to 7 to select the DMA Channel.
*@return: the enable status (ENABLE or DISABLE)
******************************************************************************/
FunctionalState DMA_GetChannelStatus(DMA_Channel_TypeDef *DMAy_Channelx)
{
	if ((uint32_t)(DMAy_Channelx) < (uint32_t)(DMA2_Channel0))
	{
        if (DMA1->ENCHSTATUS & (1UL << (((uint32_t)(DMAy_Channelx) - (uint32_t)(DMA1_Channel0)) >> 5)))
            return (ENABLE);
        else
            return (DISABLE);
	}
	else
	{
        if (DMA2->ENCHSTATUS & (1UL << (((uint32_t)(DMAy_Channelx) - (uint32_t)(DMA2_Channel0)) >> 5)))
            return (ENABLE);
        else
            return (DISABLE);
	}
}

/******************************************************************************
*@brief : Returns the status of data in FIFO of the specified DMAy Channelx.
*@param : DMAy_Channelx: where y can be 1 or 2 to select the DMA and 
*                              x can be 0 to 7 to select the DMA Channel.
*@return: The status of data in FIFO (SET or RESET).
******************************************************************************/
FlagStatus DMA_GetFIFOStatus(DMA_Channel_TypeDef *DMAy_Channelx)
{
    if (DMAy_Channelx->CHCONFIG & DMA_CHCONFIG_ACTIVE)
        return (SET);
    else
        return (RESET);
}

/******************************************************************************
*@brief : Returns the specified DMAy Channelx Flags.
*@param : DMAy_Channelx: where y can be 1 or 2 to select the DMA and 
*                              x can be 0 to 7 to select the DMA Channel.
*@return: the specifies DMAy Channelx Flags. 
*          This value can be any combination of the following values:
*            @arg DMA_FLAG_IT:   interrupt summary flag.
*            @arg DMA_FLAG_TC:   transfer complete flag
*            @arg DMA_FLAG_HTC:  half transfer complete flag
*            @arg DMA_FLAG_ERR:  transfer error flag
*            @arg DMA_FLAG_RTC:  raw transfer complete flag
*            @arg DMA_FLAG_RHTC: raw half transfer complete flag
*            @arg DMA_FLAG_RERR: raw transfer error flag
******************************************************************************/
uint32_t DMA_GetFlag(DMA_Channel_TypeDef *DMAy_Channelx)
{
    uint32_t Flags;
    uint32_t Channel;
    DMA_TypeDef *DMAy;
    
    /* calculation of the channel */
	if ((uint32_t)(DMAy_Channelx) < (uint32_t)(DMA2_Channel0))
	{
		Channel = 1UL << (((uint32_t)(DMAy_Channelx) - (uint32_t)(DMA1_Channel0)) >> 5);
		DMAy = DMA1;
	}
	else
	{
		Channel = 1UL << (((uint32_t)(DMAy_Channelx) - (uint32_t)(DMA2_Channel0)) >> 5);
		DMAy = DMA2;
	}
    
    Flags = 0;
    if (DMAy->INTTCSTATUS & Channel)
        Flags |= DMA_FLAG_TC;
    
    if (DMAy->INTTCSTATUS & (Channel << 8))
        Flags |= DMA_FLAG_HTC;
    
    if (DMAy->INTERRSTATUS & Channel)
        Flags |= DMA_FLAG_ERR;
    
    if (DMAy->INTSTATUS & Channel)
        Flags |= DMA_FLAG_IT;
    
    if (DMAy->RAWINTTCSTATUS & Channel)
        Flags |= DMA_FLAG_RTC;
    
    if (DMAy->RAWINTTCSTATUS & (Channel << 8))
        Flags |= DMA_FLAG_RHTC;
    
    if (DMAy->RAWINTERRSTATUS & Channel)
        Flags |= DMA_FLAG_RERR;
    
    return (Flags);
}

/******************************************************************************
*@brief : Returns the specified DMAy Channelx flag status.
*@param : DMAy_Channelx: where y can be 1 or 2 to select the DMA and 
*                              x can be 0 to 7 to select the DMA Channel.
*@param : Flag: specifies the flag. 
*          This parameter can be one of the following values:
*            @arg DMA_FLAG_IT:   interrupt summary flag.
*            @arg DMA_FLAG_TC:   transfer complete flag
*            @arg DMA_FLAG_HTC:  half transfer complete flag
*            @arg DMA_FLAG_ERR:  transfer error flag
*            @arg DMA_FLAG_RTC:  raw transfer complete flag
*            @arg DMA_FLAG_RHTC: raw half transfer complete flag
*            @arg DMA_FLAG_RERR: raw transfer error flag
*@return: The new state of Flag (SET or RESET).
******************************************************************************/
FlagStatus DMA_GetFlagStatus(DMA_Channel_TypeDef *DMAy_Channelx, uint32_t Flag)
{
    uint32_t Channel;
    DMA_TypeDef *DMAy;
    
    /* calculation of the channel */
	if ((uint32_t)(DMAy_Channelx) < (uint32_t)(DMA2_Channel0))
	{
		Channel = 1UL << (((uint32_t)(DMAy_Channelx) - (uint32_t)(DMA1_Channel0)) >> 5);
		DMAy = DMA1;
	}
	else
	{
		Channel = 1UL << (((uint32_t)(DMAy_Channelx) - (uint32_t)(DMA2_Channel0)) >> 5);
		DMAy = DMA2;
	}
    
    if (DMA_FLAG_TC == Flag)
    {
        if (DMAy->INTTCSTATUS & Channel)
            return (SET);
        else
            return (RESET);
    }
    
    if (DMA_FLAG_ERR == Flag)
    {
        if (DMAy->INTERRSTATUS & Channel)
            return (SET);
        else
            return (RESET);
    }
    
    if (DMA_FLAG_HTC == Flag)
    {
        if (DMAy->INTTCSTATUS & (Channel << 8))
            return (SET);
        else
            return (RESET);
    }
    
    if (DMA_FLAG_IT == Flag)
    {
        if (DMAy->INTSTATUS & Channel)
            return (SET);
        else
            return (RESET);
    }
    
    if (DMA_FLAG_RTC == Flag)
    {
        if (DMAy->RAWINTTCSTATUS & Channel)
            return (SET);
        else
            return (RESET);
    }
    
    if (DMA_FLAG_RERR == Flag)
    {
        if (DMAy->RAWINTERRSTATUS & Channel)
            return (SET);
        else
            return (RESET);
    }
    
    if (DMA_FLAG_RHTC == Flag)
    {
        if (DMAy->RAWINTTCSTATUS & (Channel << 8))
            return (SET);
        else
            return (RESET);
    }
    
    return (RESET);
}

/******************************************************************************
*@brief : Clears the specified DMAy Channelx's flags.
*@param : DMAy_Channelx: where y can be 1 or 2 to select the DMA and 
*                              x can be 0 to 7 to select the DMA Channel.
*@param : Flag: specifies the flag to clear. 
*          This parameter can be any combination of the following values:
*            @arg DMA_FLAG_IT:   interrupt summary flag.
*            @arg DMA_FLAG_TC:   transfer complete flag
*            @arg DMA_FLAG_HTC:  half transfer complete flag
*            @arg DMA_FLAG_ERR:  transfer error flag
*            @arg DMA_FLAG_RTC:  raw transfer complete flag
*            @arg DMA_FLAG_RHTC: raw half transfer complete flag
*            @arg DMA_FLAG_RERR: raw transfer error flag
*@return: None
******************************************************************************/
void DMA_ClearFlag(DMA_Channel_TypeDef *DMAy_Channelx, uint32_t Flag)
{
    uint32_t Channel;
    DMA_TypeDef *DMAy;
    
    /* calculation of the channel */
	if ((uint32_t)(DMAy_Channelx) < (uint32_t)(DMA2_Channel0))
	{
		Channel = 1UL << (((uint32_t)(DMAy_Channelx) - (uint32_t)(DMA1_Channel0)) >> 5);
		DMAy = DMA1;
	}
	else
	{
		Channel = 1UL << (((uint32_t)(DMAy_Channelx) - (uint32_t)(DMA2_Channel0)) >> 5);
		DMAy = DMA2;
	}
    
    if (Flag & (DMA_FLAG_TC | DMA_FLAG_RTC))
    {
        DMAy->INTTCCLR = Channel;
    }
    if (Flag & (DMA_FLAG_ERR | DMA_FLAG_RERR))
    {
        DMAy->INTERRCLR = Channel;
    }
    if (Flag & (DMA_FLAG_HTC | DMA_FLAG_RHTC))
    {
        DMAy->INTTCCLR = Channel << 8;
    }
    if (Flag & DMA_FLAG_IT)
    {
        DMAy->INTTCCLR = Channel | (Channel << 8);
    }
}


/******************************************************************************
*@brief : Enables or disables the specified DMAy Channelx interrupts.
*@param : DMAy_Channelx: where y can be 1 or 2 to select the DMA and 
*                              x can be 0 to 7 to select the DMA Channel.
*@param : IT: specifies the DMA interrupt sources to be enabled or disabled. 
*          This parameter can be any combination of the following values:
*            @arg DMA_IT_TC:  Transfer complete interrupt.
*            @arg DMA_IT_HTC: Half transfer complete interrupt.
*            @arg DMA_IT_ERR: Transfer error interrupt.
*@return: None
******************************************************************************/
void DMA_ITConfig(DMA_Channel_TypeDef *DMAy_Channelx, uint32_t IT, FunctionalState NewState)
{
//    uint32_t Channel;
    
    /* calculation of the channel */
	if ((uint32_t)(DMAy_Channelx) < (uint32_t)(DMA2_Channel0))
	{
//		Channel = 1UL << (((uint32_t)(DMAy_Channelx) - (uint32_t)(DMA1_Channel0)) >> 5);
	}
	else
	{
//		Channel = 1UL << (((uint32_t)(DMAy_Channelx) - (uint32_t)(DMA2_Channel0)) >> 5);
	}
    
    if (NewState != DISABLE)
    {
        DMAy_Channelx->CHCONFIG |= IT;
    }
    else
    {
        DMAy_Channelx->CHCONFIG &= ~IT;
    }
}

