
/******************************************************************************
*@file  : spl_eflash.c
*@brief : EFlash SPL module driver.
*@ver   : 1.0.0
*@date  : 2022.10.20
******************************************************************************/

#include "acm32g103_spl_conf.h" 

//#define EFLASH_BASE_FUNC_IN_SRAM 

#ifdef EFLASH_BASE_FUNC_IN_SRAM
    //ѴEFLASH_sram.c빤̣ѴӵSRAMռ
    void EFLASH_ErasePage_InSram(uint32_t addr);
    void EFLASH_ProgramWord_InSram(uint32_t addrAlign4, uint32_t dat);
#else
    //ֱӵROMд
    #define EFLASH_ErasePage_InRom     ((void (*)(uint32_t addr))(0x12000629)) 
    #define EFLASH_ProgramWord_InRom   ((void (*)(uint32_t addrAlign4, uint32_t dat))(0x1200068D))
#endif


uint32_t EFLASH_EnterCritical(void)
{
    uint32_t sr;
    sr = __get_PRIMASK();      
    __set_PRIMASK(1);
    SCB->ICSR = BIT25;   // clear systick pending bit     
    return sr;
}

void EFLASH_ExitCritical(uint32_t sr)
{
    __set_PRIMASK(sr&0x01);
}


/******************************************************************************
*@brief : erase flash page.
*@param : addr: addr in the page.
*@return: status
******************************************************************************/
EFLASH_Status EFLASH_ErasePage(uint32_t addr)
{
    uint32_t sr;
    
    assert_param( ((addr>=EFLASH_MAIN_ADDR) && (addr < EFLASH_MAIN_ADDR+EFLASH_MAIN_SIZE)) 
               || ((addr >= EFLASH_NVR_ADDR) && (addr < EFLASH_NVR_ADDR+EFLASH_NVR_SIZE)) );
      
    addr = addr & (~(EFLASH_PAGE_SIZE-1));  //page׵ַ
    
    sr = EFLASH_EnterCritical();
    EFC->CTRL &= ~(EFC_CTRL_CHIPERASEMODE | EFC_CTRL_PAGEERASEMODE | EFC_CTRL_WRITEMODE);  //֮ǰĲ
    
#ifdef EFLASH_BASE_FUNC_IN_SRAM    
    EFLASH_ErasePage_InSram(addr);  
#else  
    EFLASH_ErasePage_InRom(addr);      
#endif    
       
#if (DATA_ACCELERATE_ENABLE == 1)     
    System_InvalidateDAccelerate_by_Addr((uint32_t *)addr, EFLASH_PAGE_SIZE); 
#endif 
    
    EFLASH_ExitCritical(sr);
    
    return EFLASH_COMPLETE;

}


/******************************************************************************
*@brief : eflash ҳ
*@param : addr: ʼַ
*@param : pageNum: ҪҳĿ
*@return: status
******************************************************************************/
EFLASH_Status EFLASH_ErasePages(uint32_t addr,uint32_t pageNum)
{
    EFLASH_Status ret;
    while(pageNum--)
    {
		ret = EFLASH_ErasePage(addr);
        if(ret!=EFLASH_COMPLETE)
        {
            return ret;
        }
		addr += EFLASH_PAGE_SIZE;
    }
    
	return EFLASH_COMPLETE;
}


/*****************************************************************
@	ݣֿ֧ҳ
@	addr - ʼַ
@	len - ֽڳ
@	EFLASH_COMPLETE-ɹ ,HAL_ERROR-ʧ
******************************************************************/
EFLASH_Status EFLASH_Erase(uint32_t addr, uint32_t len)
{
    uint32_t eraseAddr;
    EFLASH_Status ret;   
  
    eraseAddr = addr &  (~(EFLASH_PAGE_SIZE-1));//ҳ׵ַ
    len += (addr &  (EFLASH_PAGE_SIZE-1));

    while(len)
    {
        ret=EFLASH_ErasePage(eraseAddr); 
        if(ret)
        {
            return ret;
        } 
        
        if(len > EFLASH_PAGE_SIZE)
        {
            len -= EFLASH_PAGE_SIZE;
        }
        else
        {
            break;
        }             
        eraseAddr += EFLASH_PAGE_SIZE;        
    }
    
    return EFLASH_COMPLETE;

}



/******************************************************************************
*@brief : program eflash word(32bit).
*@param : addrAlign4: program addr, must align 4
*@param : dat: program data.
*@return: status
******************************************************************************/
EFLASH_Status EFLASH_ProgramWord(uint32_t addrAlign4, uint32_t dat)
{   
    uint32_t sr;
    
    assert_param( ((addrAlign4>=EFLASH_MAIN_ADDR) && (addrAlign4 < EFLASH_MAIN_ADDR+EFLASH_MAIN_SIZE)) 
               || ((addrAlign4 >= EFLASH_NVR_ADDR) && (addrAlign4 < EFLASH_NVR_ADDR+EFLASH_NVR_SIZE)) );
    
    if (addrAlign4 & 0x03)
    {
        return EFLASH_ERROR_PG; 
    }
	  
    sr = EFLASH_EnterCritical();
    EFC->CTRL &= ~(EFC_CTRL_CHIPERASEMODE | EFC_CTRL_PAGEERASEMODE | EFC_CTRL_WRITEMODE);  //֮ǰĲ
    
#ifdef EFLASH_BASE_FUNC_IN_SRAM
    EFLASH_ProgramWord_InSram(addrAlign4,dat);
#else    
    EFLASH_ProgramWord_InRom(addrAlign4,dat);
#endif    

    EFLASH_ExitCritical(sr);
    
    return EFLASH_COMPLETE;

}

/******************************************************************************
*@brief : eflashWORDݡñǰȲFLASHݡ
*@param : addrAlign4:  ʼַ4ֽڶ׼
*@param : buff: ݻ棬Ҫ4ֽڶ׼ȱ4ֽڵı
*@param : wordNum: д32bit word Ŀ
*@return: status
******************************************************************************/
EFLASH_Status EFLASH_ProgramWords(uint32_t addrAlign4, void *buff, uint32_t wordNum)
{
    static volatile uint32_t count=0;
    uint32_t dat;
    uint8_t *byteptr;
    uint32_t *wordptr; 
    EFLASH_Status ret;	

    if((uint32_t)buff & 0x03)  //ָδ4ֽڶ׼
    {
        byteptr=(uint8_t *)buff;
        while(wordNum--)
        {
            memcpy(&dat,byteptr,4);
            ret = EFLASH_ProgramWord(addrAlign4,dat);          
            addrAlign4 += 4;
            byteptr += 4;        
            if(ret!=EFLASH_COMPLETE)
            {
                return ret;
            }
        }
    }
    else
    {
        wordptr=(uint32_t *)buff;
        while(wordNum--)
        {
            ret = EFLASH_ProgramWord(addrAlign4,*wordptr++);  
            addrAlign4 += 4;   
            if(ret!=EFLASH_COMPLETE)
            {
                return ret;
            } 
        }        
    }

	return EFLASH_COMPLETE; 
}

/******************************************************************************
*@brief : flashַbuffȶ4ֽڶ׼ڲԶWORDȡЧʡֽڶ
*@param : addr: ʼַ
*@param : buff: ݻ
*@param : byteNum: ҪȡֽĿ
*@return: none
******************************************************************************/
void EFLASH_Read(uint32_t addr, void *buff, uint32_t byteNum)
{
	uint32_t i,len;
	uint8_t *bptrFlash,*bptrBuff;
	uint32_t *wptrFlash,*wptrBuff;

    if((addr&0x03) || (byteNum&0x03) || ((uint32_t)buff & 0x03))   //ֽڶ
    {   
        bptrFlash = (uint8_t *)addr;
        bptrBuff =  (uint8_t *)buff;

        for (i = 0; i < byteNum; i++)
        {
            *bptrBuff++ = *bptrFlash++;
        }
    }
    else   //WORD
    {
        wptrFlash = (uint32_t *)addr;
        wptrBuff =  (uint32_t *)buff;
        
        len = byteNum >> 2;

        for (i = 0; i < len; i++)
        {
            *wptrBuff++ = *wptrFlash++;
        }         
    }
    
}

/******************************************************************************
*@brief : eflashֽݡ
*           ҳԶٱ̡
*           ڷҳԶpageݣٱ̡
*@param : addr: ʼַ
*@param : buff: ݻ棬uint8_t,uint16_t,uint32_t
*@param : byteLen: Ҫ̵ֽĿ
*@return: status
******************************************************************************/
EFLASH_Status EFLASH_BackupEraseProgram(uint32_t addr, void *buff, uint32_t byteNum)
{
    uint16_t offset; 
    uint16_t writeLen;
    uint32_t writeAddr;
    uint8_t *writePtr;    
    uint32_t  flash_buff[EFLASH_PAGE_SIZE/4];
    uint8_t *ptrBuff = (uint8_t *)buff;

    
    offset = (addr & (EFLASH_PAGE_SIZE-1));	//ҳƫ
    writeAddr = addr - offset;              //ҳʼַ

    while(byteNum>0)
    {
        //д볤
        if(byteNum <= (EFLASH_PAGE_SIZE - offset))//дݲҳʣռ䣬ûгҳΧ
        {
            writeLen = byteNum;
        }
        else
        {
            writeLen = EFLASH_PAGE_SIZE - offset;
        } 
        
        if(writeLen != EFLASH_PAGE_SIZE)//ҳд
        {
            EFLASH_Read(writeAddr, flash_buff, EFLASH_PAGE_SIZE);//ҳ
            memcpy(((uint8_t *)flash_buff)+offset,ptrBuff,writeLen);
            
            writePtr = (uint8_t *)flash_buff;
        }
        else//ҳд
        {
            writePtr = ptrBuff;
        }
        
        //Ȳд1
        if(EFLASH_ErasePage(writeAddr)!=EFLASH_COMPLETE)
        {
            return EFLASH_ERROR_PG;
        }
        
        if(EFLASH_ProgramWords(writeAddr,writePtr,EFLASH_PAGE_SIZE/4)!= EFLASH_COMPLETE)  
        {        
            return  EFLASH_ERROR_PG;
        }  
        offset = 0;				            //ƫλΪ0 
        writeAddr += EFLASH_PAGE_SIZE;	    //дַƫ        
        ptrBuff += writeLen;  	            //ָƫ
        byteNum -= writeLen;	            //ֽݼ
    }
	return EFLASH_COMPLETE; 
}


/******************************************************************************
*@brief : ȫ
*@return: status
******************************************************************************/
EFLASH_Status EFLASH_EraseSecurityCode(void)
{
    uint32_t securityCode=0xFFFFFFFF;

    return EFLASH_BackupEraseProgram(EFLASH_NVR_REMAP_ADDR,&securityCode,4);  
}

/******************************************************************************
*@brief : ֹJTAG
*@return: status
******************************************************************************/
EFLASH_Status EFLASH_DisableJtag(void)
{
    uint32_t securityCode=0x89bc3f51;
    if(EFLASH_READ_WORD(EFLASH_NVR_JTAG_DIS_2_ADDR)==securityCode)
    {
        return EFLASH_COMPLETE;
    }
    
    return EFLASH_BackupEraseProgram(EFLASH_NVR_JTAG_DIS_2_ADDR,&securityCode,4);  
}


/******************************************************************************
*@brief : ʹJTAG
*@return: status
******************************************************************************/
EFLASH_Status EFLASH_EnableJtag(void)
{
    uint32_t securityCode=0xFFFFFFFF;
    
    if(EFLASH_READ_WORD(EFLASH_NVR_JTAG_DIS_2_ADDR)==securityCode)
    {
        return EFLASH_COMPLETE;
    } 
    
    return EFLASH_BackupEraseProgram(EFLASH_NVR_JTAG_DIS_2_ADDR,&securityCode,4);  
}