/******************************************************************************
* @file      : spl_exmc.c
* @version   : 1.0
* @date      : 2023.2.8
* @brief     : EXMC SPL module driver
*   
* @history   :	
*   2023.2.8    lwq    create
*   
******************************************************************************/
#include "acm32g103_spl_conf.h" 



//#ifdef SPL_EXMC_MODULE_ENABLED

/* Default timing parameters */
const EXMC_NORSRAMTimingInitTypeDef EXMC_DefaultTimingStruct = 
{
    0x0F, /* EXMC_AddressSetupTime */
    0x0F, /* EXMC_AddressHoldTime */
    0xFF, /* EXMC_DataSetupTime */
    0x0F, /* EXMC_BusTurnAroundDuration */
    0x0F, /* EXMC_CLKDivision */
    0x0F, /* EXMC_DataLatency */
    EXMC_ACCESS_MODE_A /* EXMC_AccessMode */
};

/******************************************************************************
* @brief:  De-initializes the EXMC NOR/SRAM Banks registers to their default 
*          reset values.
* @param:  None
* @return: None
******************************************************************************/
void EXMC_NORSRAMDeInit(void)
{
    /* Release EXMC from reset state */  
    __RCC_EXMC_RESET();
}

/******************************************************************************
* @brief:  Initializes the EXMC NOR/SRAM Banks according to the specified
*          parameters in the EXMC_NORSRAMInitStruct.
* @param:  EXMC_NORSRAMInitStruct : pointer to a EXMC_NORSRAMInitTypeDef structure
*          that contains the configuration information for the EXMC NOR/SRAM 
*          specified Banks.                       
* @return: None
******************************************************************************/
void EXMC_NORSRAMInit(EXMC_NORSRAMInitTypeDef* EXMC_NORSRAMInitStruct)
{ 
    /* Check the parameters */
    assert_param(IS_EXMC_NORSRAM_BANK(EXMC_NORSRAMInitStruct->EXMC_Bank));
    assert_param(IS_EXMC_MUX(EXMC_NORSRAMInitStruct->EXMC_DataAddressMux));
    assert_param(IS_EXMC_MEMORY(EXMC_NORSRAMInitStruct->EXMC_MemoryType));
    assert_param(IS_EXMC_MEMORY_WIDTH(EXMC_NORSRAMInitStruct->EXMC_MemoryDataWidth));
    assert_param(IS_EXMC_BURSTMODE(EXMC_NORSRAMInitStruct->EXMC_BurstAccessMode));
    assert_param(IS_EXMC_WAIT_POLARITY(EXMC_NORSRAMInitStruct->EXMC_WaitSignalPolarity));
    assert_param(IS_EXMC_WRAP_MODE(EXMC_NORSRAMInitStruct->EXMC_WrapMode));
    assert_param(IS_EXMC_WAIT_SIGNAL_ACTIVE(EXMC_NORSRAMInitStruct->EXMC_WaitSignalActive));
    assert_param(IS_EXMC_WRITE_OPERATION(EXMC_NORSRAMInitStruct->EXMC_WriteOperation));
    assert_param(IS_EXMC_WAITE_SIGNAL(EXMC_NORSRAMInitStruct->EXMC_WaitSignal));
    assert_param(IS_EXMC_EXTENDED_MODE(EXMC_NORSRAMInitStruct->EXMC_ExtendedMode));
    assert_param(IS_EXMC_ASYNWAIT(EXMC_NORSRAMInitStruct->EXMC_AsynchronousWait));
    assert_param(IS_EXMC_CRAMPPAGE_SIZE(EXMC_NORSRAMInitStruct->EXMC_CRAMPageSize));
    assert_param(IS_EXMC_WRITE_BURST(EXMC_NORSRAMInitStruct->EXMC_WriteBurst));  
    assert_param(IS_EXMC_ADDRESS_SETUP_TIME(EXMC_NORSRAMInitStruct->EXMC_ReadWriteTimingStruct->EXMC_AddressSetupTime));
    assert_param(IS_EXMC_ADDRESS_HOLD_TIME(EXMC_NORSRAMInitStruct->EXMC_ReadWriteTimingStruct->EXMC_AddressHoldTime));
    assert_param(IS_EXMC_DATASETUP_TIME(EXMC_NORSRAMInitStruct->EXMC_ReadWriteTimingStruct->EXMC_DataSetupTime));
    assert_param(IS_EXMC_TURNAROUND_TIME(EXMC_NORSRAMInitStruct->EXMC_ReadWriteTimingStruct->EXMC_BusTurnAroundDuration));
    assert_param(IS_EXMC_CLK_DIV(EXMC_NORSRAMInitStruct->EXMC_ReadWriteTimingStruct->EXMC_CLKDivision));
    assert_param(IS_EXMC_DATA_LATENCY(EXMC_NORSRAMInitStruct->EXMC_ReadWriteTimingStruct->EXMC_DataLatency));
    assert_param(IS_EXMC_ACCESS_MODE(EXMC_NORSRAMInitStruct->EXMC_ReadWriteTimingStruct->EXMC_AccessMode)); 

    /* Enable EXMC clock */
    __RCC_EXMC_CLK_ENABLE();      

    /*---------------------------- EXMC SNCTL Configuration -----------------*/
    /* Clear NRBKEN, NRMUX, NRTP, NRW, NREN, SBRSTEN, NRWTPOL, WRAPEN, NRWTCFG, WREN,
            NWAITEN, EXTMODEN, ASYNCWAIT, CPS and SYNCWR bits */
    EXMC->SNCTLCFG[EXMC_NORSRAMInitStruct->EXMC_Bank] &= ((uint32_t)~( EXMC_SNCTLX_NRBKEN_Msk | EXMC_SNCTLX_NRMUX_Msk | EXMC_SNCTLX_NRTP_Msk | \
                                                            EXMC_SNCTLX_NRW_Msk | EXMC_SNCTLX_NREN_Msk | EXMC_SNCTLX_SBRSTEN_Msk | \
                                                            EXMC_SNCTLX_NRWTPOL_Msk | EXMC_SNCTLX_WRAPEN_Msk | EXMC_SNCTLX_NRWTCFG_Msk | \
                                                            EXMC_SNCTLX_WREN_Msk | EXMC_SNCTLX_NRWTEN_Msk | EXMC_SNCTLX_EXMODEN_Msk | \
                                                            EXMC_SNCTLX_ASYNCWAIT_Msk | EXMC_SNCTLX_CPS_Msk | EXMC_SNCTLX_SYNCWR_Msk));

    /* Bank1 NOR/SRAM control register configuration */ 
    EXMC->SNCTLCFG[EXMC_NORSRAMInitStruct->EXMC_Bank] |= (uint32_t)(EXMC_NORSRAMInitStruct->EXMC_DataAddressMux | \
                                                            EXMC_NORSRAMInitStruct->EXMC_MemoryType | \
                                                            EXMC_NORSRAMInitStruct->EXMC_MemoryDataWidth | \
                                                            EXMC_NORSRAMInitStruct->EXMC_BurstAccessMode | \
                                                            EXMC_NORSRAMInitStruct->EXMC_WaitSignalPolarity | \
                                                            EXMC_NORSRAMInitStruct->EXMC_WrapMode | \
                                                            EXMC_NORSRAMInitStruct->EXMC_WaitSignalActive | \
                                                            EXMC_NORSRAMInitStruct->EXMC_WriteOperation | \
                                                            EXMC_NORSRAMInitStruct->EXMC_WaitSignal | \
                                                            EXMC_NORSRAMInitStruct->EXMC_ExtendedMode | \
                                                            EXMC_NORSRAMInitStruct->EXMC_AsynchronousWait | \
                                                            EXMC_NORSRAMInitStruct->EXMC_CRAMPageSize | \
                                                            EXMC_NORSRAMInitStruct->EXMC_WriteBurst);



    if(EXMC_NORSRAMInitStruct->EXMC_MemoryType == EXMC_MEMORY_TYPE_NOR)
    {
        EXMC->SNCTLCFG[EXMC_NORSRAMInitStruct->EXMC_Bank] |= (uint32_t)EXMC_NORSRAM_FLASH_ACCESS_ENABLE;
    }

    /*---------------------------- EXMC SNTCFG Configuration -----------------*/
    /* Clear ASET, AHLD, DSET, BUSLAT, CKDIV, DLAT and ASYNCMO bits */
    EXMC->SNCTLCFG[EXMC_NORSRAMInitStruct->EXMC_Bank+1] &= ((uint32_t)~(EXMC_SNTCFGX_ASET_Msk | EXMC_SNTCFGX_AHLD_Msk | EXMC_SNTCFGX_DSET_Msk | \
                                                                EXMC_SNTCFGX_BUSLAT_Msk | EXMC_SNTCFGX_CKDIV_Msk | EXMC_SNTCFGX_DLAT_Msk | \
                                                                EXMC_SNTCFGX_ASYNCMOD_Msk));

    /* Bank1 NOR/SRAM timing register configuration */
    EXMC->SNCTLCFG[EXMC_NORSRAMInitStruct->EXMC_Bank+1] |= (uint32_t)EXMC_NORSRAMInitStruct->EXMC_ReadWriteTimingStruct->EXMC_AddressSetupTime | \
                                                            (EXMC_NORSRAMInitStruct->EXMC_ReadWriteTimingStruct->EXMC_AddressHoldTime << EXMC_SNTCFGX_AHLD_Pos) | \
                                                            (EXMC_NORSRAMInitStruct->EXMC_ReadWriteTimingStruct->EXMC_DataSetupTime << EXMC_SNTCFGX_DSET_Pos) | \
                                                            (EXMC_NORSRAMInitStruct->EXMC_ReadWriteTimingStruct->EXMC_BusTurnAroundDuration << EXMC_SNTCFGX_BUSLAT_Pos) | \
                                                            (EXMC_NORSRAMInitStruct->EXMC_ReadWriteTimingStruct->EXMC_CLKDivision << EXMC_SNTCFGX_CKDIV_Pos) | \
                                                            (EXMC_NORSRAMInitStruct->EXMC_ReadWriteTimingStruct->EXMC_DataLatency << EXMC_SNTCFGX_DLAT_Pos) | \
                                                             EXMC_NORSRAMInitStruct->EXMC_ReadWriteTimingStruct->EXMC_AccessMode;


    /*---------------------------- EXMC SNWTCFG Configuration -----------------*/
    /* Bank1 NOR/SRAM timing register for write configuration, if extended mode is used */
    if(EXMC_NORSRAMInitStruct->EXMC_ExtendedMode == EXMC_EXTENDED_MODE_ENABLE)
    {
        assert_param(IS_EXMC_ADDRESS_SETUP_TIME(EXMC_NORSRAMInitStruct->EXMC_WriteTimingStruct->EXMC_AddressSetupTime));
        assert_param(IS_EXMC_ADDRESS_HOLD_TIME(EXMC_NORSRAMInitStruct->EXMC_WriteTimingStruct->EXMC_AddressHoldTime));
        assert_param(IS_EXMC_DATASETUP_TIME(EXMC_NORSRAMInitStruct->EXMC_WriteTimingStruct->EXMC_DataSetupTime));
        assert_param(IS_EXMC_TURNAROUND_TIME(EXMC_NORSRAMInitStruct->EXMC_WriteTimingStruct->EXMC_BusTurnAroundDuration));
        assert_param(IS_EXMC_DATA_LATENCY(EXMC_NORSRAMInitStruct->EXMC_WriteTimingStruct->EXMC_DataLatency));
        assert_param(IS_EXMC_ACCESS_MODE(EXMC_NORSRAMInitStruct->EXMC_WriteTimingStruct->EXMC_AccessMode));

        /* Clear WASET, WAHLD, WDSET, WBUSLAT, DLAT, and WASYNCMOD bits */
        EXMC_Ex->SNWTCFG[EXMC_NORSRAMInitStruct->EXMC_Bank] &= ((uint32_t)~(EXMC_SNWTCFGX_WASET_Msk | EXMC_SNWTCFGX_WAHLD_Msk | EXMC_SNWTCFGX_WDSET_Msk | \
                                                                EXMC_SNWTCFGX_WBUSLAT_Msk | EXMC_SNWTCFGX_DLAT_Msk | EXMC_SNWTCFGX_WASYNCMOD_Msk));

        EXMC_Ex->SNWTCFG[EXMC_NORSRAMInitStruct->EXMC_Bank] |= (uint32_t)EXMC_NORSRAMInitStruct->EXMC_WriteTimingStruct->EXMC_AddressSetupTime |
                                                                (EXMC_NORSRAMInitStruct->EXMC_WriteTimingStruct->EXMC_AddressHoldTime << EXMC_SNWTCFGX_WAHLD_Pos )|
                                                                (EXMC_NORSRAMInitStruct->EXMC_WriteTimingStruct->EXMC_DataSetupTime << EXMC_SNWTCFGX_WDSET_Pos) |
                                                                (EXMC_NORSRAMInitStruct->EXMC_WriteTimingStruct->EXMC_BusTurnAroundDuration << EXMC_SNWTCFGX_WBUSLAT_Pos) |
                                                                (EXMC_NORSRAMInitStruct->EXMC_WriteTimingStruct->EXMC_DataLatency << EXMC_SNWTCFGX_DLAT_Pos) |
                                                                 EXMC_NORSRAMInitStruct->EXMC_WriteTimingStruct->EXMC_AccessMode;

    }
    else
    {
        EXMC_Ex->SNWTCFG[EXMC_NORSRAMInitStruct->EXMC_Bank] = 0x0FFFFFFF;
    }
}

/******************************************************************************
* @brief:  Fills each EXMC_NORSRAMInitStruct member with its default value.
* @param:  EXMC_NORSRAMInitStruct: pointer to a EXMC_NORSRAMInitTypeDef structure 
*          which will be initialized.
* @return: None
******************************************************************************/
void EXMC_NORSRAMStructInit(EXMC_NORSRAMInitTypeDef* EXMC_NORSRAMInitStruct)
{  
    /* Reset NOR/SRAM Init structure parameters values */
    EXMC_NORSRAMInitStruct->EXMC_Bank = EXMC_NORSRAM_BANK1;
    EXMC_NORSRAMInitStruct->EXMC_DataAddressMux = EXMC_DATA_ADDRESS_MUX_ENABLE;
    EXMC_NORSRAMInitStruct->EXMC_MemoryType = EXMC_MEMORY_TYPE_SRAM;
    EXMC_NORSRAMInitStruct->EXMC_MemoryDataWidth = EXMC_NORSRAM_MEM_BUS_WIDTH_8;
    EXMC_NORSRAMInitStruct->EXMC_BurstAccessMode = EXMC_NORSRAM_FLASH_ACCESS_DISABLE;
    EXMC_NORSRAMInitStruct->EXMC_WaitSignalPolarity = EXMC_WAIT_SIGNAL_POLARITY_LOW;
    EXMC_NORSRAMInitStruct->EXMC_WrapMode = EXMC_WRAP_MODE_DISABLE;
    EXMC_NORSRAMInitStruct->EXMC_WaitSignalActive = EXMC_WAITSIGNALACTIVE_BEFOREWAITSTATE;
    EXMC_NORSRAMInitStruct->EXMC_WriteOperation = EXMC_WRITEOPERATION_ENABLE;
    EXMC_NORSRAMInitStruct->EXMC_WaitSignal = EXMC_WAIT_SIGNAL_ENABLE;
    EXMC_NORSRAMInitStruct->EXMC_ExtendedMode = EXMC_EXTENDED_MODE_DISABLE;
    EXMC_NORSRAMInitStruct->EXMC_AsynchronousWait = EXMC_ASYNCHRONOUS_WAIT_DISABLE;
    EXMC_NORSRAMInitStruct->EXMC_CRAMPageSize = EXMC_CRAMPAGE_SIZE_AUTO;
    EXMC_NORSRAMInitStruct->EXMC_WriteBurst = EXMC_WRITE_BURST_DISABLE;
    EXMC_NORSRAMInitStruct->EXMC_ReadWriteTimingStruct = (EXMC_NORSRAMTimingInitTypeDef*)((uint32_t)&EXMC_DefaultTimingStruct);
    EXMC_NORSRAMInitStruct->EXMC_WriteTimingStruct = (EXMC_NORSRAMTimingInitTypeDef*)((uint32_t)&EXMC_DefaultTimingStruct);
}

/******************************************************************************
* @brief:  Enables or disables the specified NOR/SRAM Memory Bank.
* @param:  EXMC_Bank: specifies the EXMC Bank to be used.
* @param:  NewState: new state of the EXMC_Bank. This parameter can be: ENABLE or DISABLE.
* @return: None
******************************************************************************/
void EXMC_NORSRAMCmd(uint32_t EXMC_Bank, FunctionalState NewState)
{
    assert_param(IS_EXMC_NORSRAM_BANK(EXMC_Bank));
    assert_param(IS_FUNCTIONAL_STATE(NewState));

    if (NewState != DISABLE)
    {
        /* Enable the selected NOR/SRAM Bank by setting the NRBKEN bit in the SNCTLx register */
        EXMC->SNCTLCFG[EXMC_Bank] |= EXMC_SNCTLX_NRBKEN;
    }
    else
    {
        /* Disable the selected NOR/SRAM Bank by clearing the NRBKEN bit in the SNCTLx register */
        EXMC->SNCTLCFG[EXMC_Bank] &= (~EXMC_SNCTLX_NRBKEN);
    }
}












//#endif    /* SPL_EXMC_MODULE_ENABLED */


/************************ (C) COPYRIGHT AISINOCHIP *****END OF FILE****/


