
/******************************************************************************
*@file  : spl_iwdt.c
*@brief : IWDT SPL module driver.
******************************************************************************/

#include "acm32g103_spl_conf.h" 


/******************************************************************************
*@brief : Enables or disables write access to IWDT_ PR, IWDT_ RLR, 
*         IWDT_ WINR, IWDT_ WUTR register.
*@param   NewState: specifies the new state of write access.
*                   This parameter can be: ENABLE or DISABLE.
*@return: None
******************************************************************************/
void IWDT_WriteAccessCmd(FunctionalState NewState)
{
    /* Check the parameters */
    assert_param(IS_FUNCTIONAL_STATE(NewState));
    
    if (NewState != DISABLE)
        IWDT->CMDR = IWDT_CMD_WRITE_ENABLE;
    else
        IWDT->CMDR = IWDT_CMD_WRITE_DISABLE;
}

/******************************************************************************
*@brief : Enables or disables wakeup from STOP mode.
*@param   NewState: specifies the new state of wakeup.
*                   This parameter can be: ENABLE or DISABLE.
*@return: None
******************************************************************************/
void IWDT_WakeupCmd(FunctionalState NewState)
{
    /* Check the parameters */
    assert_param(IS_FUNCTIONAL_STATE(NewState));
    
    if (NewState != DISABLE)
        IWDT->CMDR = IWDT_CMD_WAKEUP_ENABLE;
    else
        IWDT->CMDR = IWDT_CMD_WAKEUP_DISABLE;
}

/******************************************************************************
*@brief : Enables IWDT.
*@param   None
*@return: None
******************************************************************************/
void IWDT_Enable(void)
{
    IWDT->CMDR = IWDT_CMD_ENABLE;
}

/******************************************************************************
*@brief : Reloads IWDT counter with value defined in the reload register.
*@param   None
*@return: None
******************************************************************************/
void IWDT_ReloadCounter(void)
{
    IWDT->CMDR = IWDT_CMD_RELOAD;
}

/******************************************************************************
*@brief : Set the IWDT prescaler.
*@param : Prescaler: specifies the IWDG Prescaler value.
*                    This parameter can be one of @ref IWDT_Clock_Prescaler.
*                      @arg IWDT_CLOCK_PRESCALER_4: IWDG prescaler set to 4.
*                      @arg IWDT_CLOCK_PRESCALER_8: IWDG prescaler set to 8.
*                      @arg IWDT_CLOCK_PRESCALER_16: IWDG prescaler set to 16.
*                      @arg IWDT_CLOCK_PRESCALER_32: IWDG prescaler set to 32.
*                      @arg IWDT_CLOCK_PRESCALER_64: IWDG prescaler set to 64.
*                      @arg IWDT_CLOCK_PRESCALER_128: IWDG prescaler set to 128.
*                      @arg IWDT_CLOCK_PRESCALER_256: IWDG prescaler set to 256.
*@return: None
******************************************************************************/
void IWDT_SetPrescaler(uint32_t Prescaler)
{
    /* Check the parameters */
    assert_param(IS_IWDT_PRESCALER(Prescaler));
    
    IWDT->PR = (IWDT->PR & ~IWDT_PR_PR) | Prescaler;
}

/******************************************************************************
*@brief : Set the IWDT Reload value.
*@param : Reload: specifies the IWDG Reload value.
*                 This parameter must be a number between 0 and 0x0FFF.
*@return: None
******************************************************************************/
void IWDG_SetReload(uint32_t Reload)
{
    /* Check the parameters */
    assert_param(IS_IWDT_RELOAD(Reload));
    
    IWDT->RLR = (IWDT->RLR & ~IWDT_RLR_PL) | Reload;
}

/******************************************************************************
*@brief : Set the IWDT window value.
*@param : Reload: specifies the IWDG window value.
*                 This parameter must be a number between 0 and 0x0FFF.
*@return: None
******************************************************************************/
void IWDG_SetWindow(uint32_t Window)
{
    /* Check the parameters */
    assert_param(IS_IWDT_WINDOW(Window));
    
    IWDT->WINR = (IWDT->WINR & ~IWDT_WINR_WIN) | Window;
}

/******************************************************************************
*@brief : Set the IWDT wakeup value.
*@param : Reload: specifies the IWDG wakeup value.
*                 This parameter must be a number between 0 and 0x0FFF.
*@return: None
******************************************************************************/
void IWDG_SetWakeup(uint32_t Wakeup)
{
    /* Check the parameters */
    assert_param(IS_IWDT_WAKEUP(Wakeup));
    
    IWDT->WUTR = (IWDT->WUTR & ~IWDT_WUTR_WUT) | Wakeup;
}

/******************************************************************************
*@brief : Checks whether the specified IWDT flag is set or not.
*@param : Flag: specifies the flag to check.
*               This parameter can be one of @ref IWDT_Flag.
*                 @arg IWDT_FLAG_PVU: Prescaler value update.
*                 @arg IWDT_FLAG_RVU: Reload value update.
*                 @arg IWDT_FLAG_WVU: Window value update.
*                 @arg IWDT_FLAG_WTU: Wakeup value update.
*                 @arg IWDT_FLAG_RLF: Dog feeding completed.
*@return: The new state of IWDT_Flag (SET or RESET).
******************************************************************************/
FlagStatus IWDT_GetFlagStatus(uint32_t Flag)
{
    /* Check the parameters */
    assert_param(IS_IWDT_FLAGS(Flag));

    if (IWDT->SR & Flag)
        return (SET);
    else
        return (RESET);
}


