
/******************************************************************************
*@file  : spl_lptim.c
*@brief : LPTIM SPL module driver.
******************************************************************************/

#include "acm32g103_spl_conf.h" 



/******************************************************************************
*@brief : Deinitializes the LPTIMx peripheral registers to their default reset values.
*@param : LPTIMx: where x can be 1.
*@return: None.
******************************************************************************/
void LPTIM_DeInit(LPTIM_TypeDef *LPTIMx)
{
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));

    if (LPTIMx == LPTIM1)
    {
        __RCC_LPTIM1_CLK_ENABLE();
        __RCC_LPTIM1_CLK_DISABLE();
    }
}

/******************************************************************************
*@brief : Reset the LPTIMx peripheral registers to their default reset values.
*@param : LPTIMx: where x can be 1.
*@return: None.
******************************************************************************/
void LPTIM_Reset(LPTIM_TypeDef *LPTIMx)
{
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));

    if (LPTIMx == LPTIM1)
    {
        __RCC_LPTIM1_RESET();
    }
}

/******************************************************************************
*@brief : Initializes the LPTIMx peripheral according to the specified parameters
*         in the LPTIM_InitStruct.
*@param : LPTIMx: where x can be 1.
*@param : LPTIM_InitStruct: pointer to an LPTIM_InitTypeDef structure that 
*                           contains the configuration information for the 
*                           specified LPTIM peripheral.
*@return: None
******************************************************************************/
void LPTIM_Init(LPTIM_TypeDef *LPTIMx, LPTIM_InitTypeDef *LPTIM_InitStruct)
{
    uint32_t temp;

    /* Check the parameters */
    
    /* LPTIM Reset */
    LPTIM_Reset(LPTIMx);
    
    temp = LPTIMx->CFGR1;
    
    temp &= ~(LPTIM_CFGR1_WAVPOL | LPTIM_CFGR1_WAVE | LPTIM_CFGR1_PRESC | LPTIM_CFGR1_CKSEL);
    
    temp |= LPTIM_InitStruct->OutputPolarity | LPTIM_InitStruct->Waveform | LPTIM_InitStruct->Prescaler;
    
    if (LPTIM_InitStruct->ClockSource == LPTIM_CLOCKSOURCE_EXTERNAL_INPUT1)
    {
        temp |= LPTIM_CFGR1_CKSEL;
    }
    else if (LPTIMx == LPTIM1)
    {
        RCC_LPTIM1CLKConfig(LPTIM_InitStruct->ClockSource);
    }
    LPTIMx->CFGR1 = temp;
}


/******************************************************************************
*@brief : Fills each DMA_InitStruct member with its default value.
*@param : DMA_InitStruct: pointer to a DMA_InitTypeDef structure that contains
*         the configuration information for the specified DMA Channel.
*@return: None
******************************************************************************/
void LPTIM_StructInit(LPTIM_InitTypeDef *LPTIM_InitStruct)
{
    LPTIM_InitStruct->ClockSource       = LPTIM_CLOCKSOURCE_INTERNAL_PCLK;
    LPTIM_InitStruct->Prescaler         = LPTIM_CLOCKPRESCALER_DIV1;
    LPTIM_InitStruct->Waveform          = LPTIM_WAVEFORM_PWM_ONEPULSE;
    LPTIM_InitStruct->OutputPolarity    = LPTIM_OUTPUTPOLARITY_HIGH;
}

/******************************************************************************
*@brief : Enables or disables the specified LPTIM peripheral.
*@param : LPTIMx: where x can be 1.
*@param : NewState: new state of the LPTIM.
*                   This parameter can be: ENABLE or DISABLE.
*@return: None
******************************************************************************/
void LPTIM_Cmd(LPTIM_TypeDef* LPTIMx, FunctionalState NewState)
{
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));
    assert_param(IS_LPTIM_FUNCTIONAL_STATE(NewState));
    
    if (NewState != DISABLE)
        LPTIMx->CR |= LPTIM_CR_ENABLE;
    else
        LPTIMx->CR &= ~LPTIM_CR_ENABLE;
}

/******************************************************************************
*@brief : Selects the Clock source of the LPTIM counter.
*@param : LPTIMx: where x can be 1.
*@param : ClockSource: the selected clock source.
*           This parameter can be one of @ref LPUART_ClockSource.
*             @arg LPTIM_CLOCKSOURCE_INTERNAL_PCLK: specifies RC32K as the clock source.
*             @arg LPTIM_CLOCKSOURCE_INTERNAL_RC32K: specifies XTL as the clock source.
*             @arg LPTIM_CLOCKSOURCE_INTERNAL_RC64M: specifies the 4 division of PCLK as the clock source.
*             @arg LPTIM_CLOCKSOURCE_INTERNAL_XTL: specifies the 8 division of PCLK as the clock source.
*             @arg LPTIM_CLOCKSOURCE_EXTERNAL_INPUT1: specifies the 16 division of PCLK as the clock source.
*@return: None
******************************************************************************/
void LPTIM_SelectClockSource(LPTIM_TypeDef* LPTIMx, uint32_t ClockSource)
{  
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));
    assert_param(IS_LPTIM_CLOCK_SOURCE(ClockSource));
    
    /* set the clock source */
    if (LPTIMx == LPTIM1)
    {
        if (ClockSource == LPTIM_CLOCKSOURCE_EXTERNAL_INPUT1)
        {
            LPTIMx->CFGR1 |= LPTIM_CFGR1_CKSEL;
        }
        else
        {
            RCC_LPUART1CLKConfig(ClockSource);
            LPTIMx->CFGR1 &= ~LPTIM_CFGR1_CKSEL;
        }
    }
}

/******************************************************************************
*@brief : Configures the polarity of the edge to be used to count
*         if the ULPTIM input is selected.
*@param : ClockPolarity: the selected clock polarity.
*                        This parameter can be one of @ref LPTIM_CounterPolarity.
*                          @arg LPTIM_COUNTERPOLARITY_RISING: The rising edge.
*                          @arg LPTIM_COUNTERPOLARITY_FALLING: The falling edge.
*                          @arg LPTIM_COUNTERPOLARITY_RISING_FALLING: Both rising and falling edges.
*@return: None
******************************************************************************/
void LPTIM_SelectULPTIMClockPolarity(LPTIM_TypeDef* LPTIMx, uint32_t ClockPolarity)
{
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));
    assert_param(IS_LPTIM_CLOCK_POLARITY(ClockPolarity));
    
    LPTIMx->CFGR1 = (LPTIMx->CFGR1 & ~LPTIM_CFGR1_CKPOL) | ClockPolarity;
}

/******************************************************************************
*@brief : Configures the Clock Prescaler.
*@param : Prescaler: the selected clock prescaler.
*                    This parameter can be one of @ref LPTIM_ClockPrescaler.
*                      @arg LPTIM_CLOCKPRESCALER_DIV1: clock prescaler set to 1.
*                      @arg LPTIM_CLOCKPRESCALER_DIV2: clock prescaler set to 2.
*                      @arg LPTIM_CLOCKPRESCALER_DIV4: clock prescaler set to 4.
*                      @arg LPTIM_CLOCKPRESCALER_DIV8: clock prescaler set to 8.
*                      @arg LPTIM_CLOCKPRESCALER_DIV16: clock prescaler set to 16.
*                      @arg LPTIM_CLOCKPRESCALER_DIV32: clock prescaler set to 32.
*                      @arg LPTIM_CLOCKPRESCALER_DIV64: clock prescaler set to 64.
*                      @arg LPTIM_CLOCKPRESCALER_DIV128: clock prescaler set to 128.
*@return: None
******************************************************************************/
void LPTIM_ConfigPrescaler(LPTIM_TypeDef* LPTIMx, uint32_t Prescaler)
{
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));
    assert_param(IS_LPTIM_CLOCK_PRESCALER(Prescaler));
    
    LPTIMx->CFGR1 = (LPTIMx->CFGR1 & ~LPTIM_CFGR1_PRESC) | Prescaler;
}

/******************************************************************************
*@brief : Configures the LPTIM Output pin polarity..
*@param : OutputPolarity: Specifies the LPTIM Output pin polarity.
*                         This parameter can be one of @ref LPTIM_OutputPolarity.
*                           @arg LPTIM_OUTPUTPOLARITY_HIGH: output high level.
*                           @arg LPTIM_OUTPUTPOLARITY_LOW: output low level.
*@return: None
******************************************************************************/
void LPTIM_ConfigOutputPolarity(LPTIM_TypeDef* LPTIMx, uint32_t OutputPolarity)
{
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));
    assert_param(IS_LPTIM_OUTPUT_POLARITY(OutputPolarity));
    
    LPTIMx->CFGR1 = (LPTIMx->CFGR1 & ~LPTIM_CFGR1_WAVPOL) | OutputPolarity;
}

/******************************************************************************
*@brief : Selects the trigger source for the counter and its polarity, filter.
*@param : LPTIMx: where x can be 1.
*@param : Source: specifies the trigger source of LPTIM trigger event.
*                   This parameter can be one of @ref LPTIM_TriggerSource.
*                     @arg LPTIM_TRIGSOURCE_0: external trig0.
*                     @arg LPTIM_TRIGSOURCE_1: external trig1.
*                     @arg LPTIM_TRIGSOURCE_2: external trig2.
*                     @arg LPTIM_TRIGSOURCE_3: external trig3.
*                     @arg LPTIM_TRIGSOURCE_4: external trig4.
*                     @arg LPTIM_TRIGSOURCE_5: external trig5.
*                     @arg LPTIM_TRIGSOURCE_6: external trig6.
*                     @arg LPTIM_TRIGSOURCE_7: external trig7.
*@param : Polarity: specifies the trigger signal Polarity.
*                   This parameter can be one of @ref LPTIM_TriggerPolarity.
*                     @arg LPTIM_TRIGPOLARITY_RISING: The rising edge.
*                     @arg LPTIM_TRIGPOLARITY_FALLING: The falling edge.
*                     @arg LPTIM_TRIGPOLARITY_RISING_FALLING: Both rising and falling edges.
*@return: None
******************************************************************************/
void LPTIM_ConfigExternalTrigger(LPTIM_TypeDef* LPTIMx, uint32_t Source, uint32_t Polarity)
{
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));
    assert_param(Source != LPTIM_TRIGSOURCE_SOFTWARE);
    assert_param(IS_LPTIM_TRIG_SOURCE(Source));
    assert_param(IS_LPTIM_TRIG_POLARITY(Polarity));
    
    LPTIMx->CFGR1 = (LPTIMx->CFGR1 & ~(LPTIM_CFGR1_TRIGEN | LPTIM_CFGR1_TRIGSEL)) \
                    | Source | Polarity;
}



/******************************************************************************
*@brief : Selects a software start of the counter.
*@param : LPTIMx: where x can be 1.
*@return: None
******************************************************************************/
void LPTIM_SelectSoftwareStart(LPTIM_TypeDef* LPTIMx)
{
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));
    
    LPTIMx->CFGR1 &= ~LPTIM_CFGR1_TRIGEN;
}

/******************************************************************************
*@brief : Selects the trigger source for the counter and its polarity, filter.
*@param : LPTIMx: where x can be 1.
*@param : Filter: specifies the trigger signal Filter.
*                 This parameter can be one of @ref LPTIM_TriggerFilter.
*                   @arg LPTIM_TRIGFILTER_1CLK: Filter 1 cycles.
*                   @arg LPTIM_TRIGFILTER_2CLK: Filter 2 cycles.
*                   @arg LPTIM_TRIGFILTER_4CLK: Filter 4 cycles.
*                   @arg LPTIM_TRIGFILTER_8CLK: Filter 8 cycles.
*@return: None
******************************************************************************/
void LPTIM_ConfigTriggerGlitchFilter(LPTIM_TypeDef* LPTIMx, uint32_t Filter)
{
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));
    assert_param(IS_LPTIM_TRIG_FILTER(Filter));
    
    LPTIMx->CFGR1 = (LPTIMx->CFGR1 & ~LPTIM_CFGR1_TRIGFLT) | Filter;
}

/******************************************************************************
*@brief : Configures the digital filter for the external clock by determining the number
*         of consecutive samples at the specified level to detect a correct transition.
*@param : Filter: the number of samples to detect a valid transition.
*                 This parameter can be one of @ref LPTIM_CounterFilter.
*                   @arg LPTIM_COUNTERFILTER_1CLK: Filter 1 cycles.
*                   @arg LPTIM_COUNTERFILTER_2CLK: Filter 2 cycles.
*                   @arg LPTIM_COUNTERFILTER_4CLK: Filter 4 cycles.
*                   @arg LPTIM_COUNTERFILTER_8CLK: Filter 8 cycles.
*@return: None
******************************************************************************/
void LPTIM_ConfigClockGlitchFilter(LPTIM_TypeDef* LPTIMx, uint32_t Filter)
{
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));
    assert_param(IS_LPTIM_COUNTER_FILTER(Filter));
    
    LPTIMx->CFGR1 = (LPTIMx->CFGR1 & ~LPTIM_CFGR1_CKFLT) | Filter;
}

/******************************************************************************
*@brief : Selects an operating mode.
*@param : LPTIMx: where x can be 1.
*@param : Mode: the selected mode.
*               This parameter can be one of @ref LPTIM_Mode.
*                 @arg LPTIM_MODE_CONTINUOUS: Timer starts in Continuous mode.
*                 @arg LPTIM_MODE_SINGLE: Timer will starts in Single mode.
*@return: None
******************************************************************************/
void LPTIM_SelectOperatingMode(LPTIM_TypeDef* LPTIMx, uint32_t Mode)
{
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));
    assert_param(IS_LPTIM_MODE(Mode));
    
    LPTIMx->CR |= Mode;
}

/******************************************************************************
*@brief : Enables or disables the Timeout function.
*@param : LPTIMx: where x can be 1.
*@param : NewState: new state of the Timeout function.
*                   This parameter can be: ENABLE or DISABLE.
*@return: None
******************************************************************************/
void LPTIM_TimoutCmd(LPTIM_TypeDef* LPTIMx, FunctionalState NewState)
{
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));
    assert_param(IS_LPTIM_FUNCTIONAL_STATE(NewState));
    
    if (NewState != DISABLE)
        LPTIMx->CFGR1 |= LPTIM_CFGR1_TIMOUT;
    else
        LPTIMx->CFGR1 &= ~LPTIM_CFGR1_TIMOUT;
}

/******************************************************************************
*@brief : Configures the Waveform shape.
*@param : LPTIMx: where x can be 1.
*@param : Waveform: the selected waveform shape.
*                   This parameter can be one of @ref LPTIM_Waveform.
*                     @arg LPTIM_WAVEFORM_PWM_ONEPULSE: PWM/One Pulse is selected.
*                     @arg LPTIM_WAVEFORM_SETONCE: Set once is selected.
*@return: None
******************************************************************************/
void LPTIM_ConfigWaveform(LPTIM_TypeDef* LPTIMx, uint32_t Waveform)
{
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));
    assert_param(IS_LPTIM_WAVEFORM(Waveform));
    
    LPTIMx->CFGR1 = (LPTIMx->CFGR1 & ~LPTIM_CFGR1_WAVE) | Waveform;
}

/******************************************************************************
*@brief : Configures the Autoreload and Compare registers update mode.
*@param : LPTIMx: where x can be 1.
*@param : UpdateMode: The selected update mode.
*                     This parameter can be one of @ref LPTIM_UpdatingMode.
*                       @arg LPTIM_UPDATE_IMMEDIATE: Registers updated after APB write.
*                       @arg LPTIM_UPDATE_ENDOFPERIOD: Registers updated at the end of current timer preload.
*@return: None
******************************************************************************/
void LPTIM_ConfigUpdate(LPTIM_TypeDef* LPTIMx, uint32_t UpdateMode)
{
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));
    assert_param(IS_LPTIM_UPDATE_MODE(UpdateMode));
    
    LPTIMx->CFGR1 = (LPTIMx->CFGR1 & ~LPTIM_CFGR1_PRELOAD) | UpdateMode;
}

/******************************************************************************
*@brief : Writes the passed parameter in the Autoreload register.
*@param : LPTIMx: where x can be 1.
*@param : Autoreload: The Autoreload value.
*                     This parameter must be a value between 0x0001 and 0xFFFF.
*@return: None
******************************************************************************/
void LPTIM_SetAutoreloadValue(LPTIM_TypeDef* LPTIMx, uint32_t Autoreload)
{
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));
    assert_param(IS_LPTIM_AUTORELOAD(Autoreload));
    
    LPTIMx->ARR = (LPTIMx->ARR & ~LPTIM_ARR_ARR) | Autoreload;
}

/******************************************************************************
*@brief : Writes the passed parameter in the Compare register.
*@param : LPTIMx: where x can be 1.
*@param : Compare: The Compare value.
*                  This parameter must be a value between 0x0000 and 0xFFFE.
*@return: None
******************************************************************************/
void LPTIM_SetCompareValue(LPTIM_TypeDef* LPTIMx, uint32_t Compare)
{
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));
    assert_param(IS_LPTIM_COMPARE(Compare));
    
    LPTIMx->CMP = (LPTIMx->CMP & ~LPTIM_CMP_CMP) | Compare;
}

/******************************************************************************
*@brief : Writes the passed parameter in the Repetition Counter register.
*@param : LPTIMx: where x can be 1.
*@param : RepetitionCounter: The Repetition Counter value.
*                  This parameter must be a value between 0x00 and 0xFF.
*@return: None
******************************************************************************/
void LPTIM_SetRepetitionCounterValue(LPTIM_TypeDef* LPTIMx, uint32_t RepetitionCounter)
{
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));
    assert_param(IS_LPTIM_REPETITION_COUNTER(RepetitionCounter));
    
    LPTIMx->RCR = (LPTIMx->RCR & ~LPTIM_RCR_REP) | RepetitionCounter;
}

/******************************************************************************
*@brief : Enables or disables the Counter mode. When the Counter mode is enabled,
*         the counter is incremented each valid event on ULPTIM external Input1.
*@param : LPTIMx: where x can be 1.
*@param : NewState: new state of the Counter mode.
*                   This parameter can be: ENABLE or DISABLE.
*@return: None
******************************************************************************/
void LPTIM_SelectCounterMode(LPTIM_TypeDef* LPTIMx, FunctionalState NewState)
{
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));
    assert_param(IS_LPTIM_FUNCTIONAL_STATE(NewState));
    
    if (NewState != DISABLE)
        LPTIMx->CFGR1 |= LPTIM_CFGR1_COUNTMODE;
    else
        LPTIMx->CFGR1 &= ~LPTIM_CFGR1_COUNTMODE;
}

/******************************************************************************
*@brief : Enables or disables the Encoder mode.
*@param : LPTIMx: where x can be 1.
*@param : NewState: new state of the Encoder mode.
*                   This parameter can be: ENABLE or DISABLE.
*@return: None
******************************************************************************/
void LPTIM_SelectEncoderMode(LPTIM_TypeDef* LPTIMx, FunctionalState NewState)
{
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));
    assert_param(IS_LPTIM_FUNCTIONAL_STATE(NewState));
    
    if (NewState != DISABLE)
        LPTIMx->CFGR1 |= LPTIM_CFGR1_ENC;
    else
        LPTIMx->CFGR1 &= ~LPTIM_CFGR1_ENC;
}

/******************************************************************************
*@brief : Gets the LPTIMx counter value.
*@param : LPTIMx: where x can be 1.
*@return: Counter Register value.
******************************************************************************/
uint32_t LPTIM_GetCounterValue(LPTIM_TypeDef* LPTIMx)
{
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));
    
    return (LPTIMx->CNT);
}

/******************************************************************************
*@brief : Gets the LPTIMx Autoreload value.
*@param : LPTIMx: where x can be 1.
*@return: Autoreload Register value.
******************************************************************************/
uint32_t LPTIM_GetAutoreloadValue(LPTIM_TypeDef* LPTIMx)
{
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));
    
    return (LPTIMx->ARR);
}

/******************************************************************************
*@brief : Gets the LPTIMx Compare value.
*@param : LPTIMx: where x can be 1.
*@return: Compare Register value.
******************************************************************************/
uint32_t LPTIM_GetCompareValue(LPTIM_TypeDef* LPTIMx)
{
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));
    
    return (LPTIMx->CMP);
}

/******************************************************************************
*@brief : Gets the LPTIMx Repetition Couter value.
*@param : LPTIMx: where x can be 1.
*@return: Repetition Couter Register value.
******************************************************************************/
uint32_t LPTIM_GetRepetitionCounterValue(LPTIM_TypeDef* LPTIMx)
{
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));
    
    return (LPTIMx->RCR);
}

/******************************************************************************
*@brief : LPTIM Input 1 Remap.
*@param : LPTIMx: where x can be 1.
*@param : Input1: Input1 source.
*                 This parameter can be one of @ref LPTIM_Input1_Source.
*                   @arg LPTIM_INPUT1SOURCE_0: Input1 source 0.
*                   @arg LPTIM_INPUT1SOURCE_1: Input1 source 1.
*                   @arg LPTIM_INPUT1SOURCE_2: Input1 source 2.
*                   @arg LPTIM_INPUT1SOURCE_3: Input1 source 3.
*@return: None
******************************************************************************/
void LPTIM_Input1RemapConfig(LPTIM_TypeDef* LPTIMx, uint32_t Input1)
{
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));
    assert_param(IS_LPTIM_INPUT1_SOURCE(Input1));
    
    LPTIMx->CFGR2 = (LPTIMx->CFGR2 & ~LPTIM_CFGR2_IN1SEL) | Input1;
}

/******************************************************************************
*@brief : LPTIM Input 2 Remap.
*@param : LPTIMx: where x can be 1.
*@param : Input2: Input2 source.
*                 This parameter can be one of @ref Input2.
*                   @arg LPTIM_INPUT2SOURCE_0: Input2 source 0.
*                   @arg LPTIM_INPUT2SOURCE_1: Input2 source 1.
*                   @arg LPTIM_INPUT2SOURCE_2: Input2 source 2.
*                   @arg LPTIM_INPUT2SOURCE_3: Input2 source 3.
*@return: None
******************************************************************************/
void LPTIM_Input2RemapConfig(LPTIM_TypeDef* LPTIMx, uint32_t Input2)
{
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));
    assert_param(IS_LPTIM_INPUT2_SOURCE(Input2));
    
    LPTIMx->CFGR2 = (LPTIMx->CFGR2 & ~LPTIM_CFGR2_IN2SEL) | Input2;
}

/******************************************************************************
*@brief : Enables or disables Reset Counter After Read Counter operation.
*@param : LPTIMx: where x can be 1.
*@param : NewState: new state of the Reset Counter After Read Counter function.
*                   This parameter can be: ENABLE or DISABLE.
*@return: None
******************************************************************************/
void LPTIM_ResetCounterAfterReadCounter(LPTIM_TypeDef* LPTIMx, FunctionalState NewState)
{
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));
    assert_param(IS_LPTIM_FUNCTIONAL_STATE(NewState));
    
    if (NewState != DISABLE)
        LPTIMx->CR |= LPTIM_CR_RSTARE;
    else
        LPTIMx->CR &= ~LPTIM_CR_RSTARE;
}

/******************************************************************************
*@brief : Reset Counter.
*         COUNTRST cannot be set to "1" by software before it has been cleared 
*         by hardware. Therefore, the software should check whether the COUNTRST 
*         position has been cleared before attempting to set it to "1".
*@param : LPTIMx: where x can be 1.
*@return: None
******************************************************************************/
void LPTIM_ResetCounter(LPTIM_TypeDef* LPTIMx)
{
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));
    
    LPTIMx->CR |= LPTIM_CR_COUNTRST;
}

/******************************************************************************
*@brief : Get COUNTRST state.
*@param : LPTIMx: where x can be 1.
*@return: COUNTRST state
******************************************************************************/
FlagStatus LPTIM_GetResetCounterState(LPTIM_TypeDef* LPTIMx)
{
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));
    
    if (LPTIMx->CR & LPTIM_CR_COUNTRST)
        return (SET);
    else
        return (RESET);
}

/******************************************************************************
*@brief : Enables or disables the specified LPTIM interrupts.
*@param : LPTIMx: where x can be 1.
*@param : IT: specifies the LPTIM interrupts sources to be enabled or disabled.
*             This parameter can be combination of @ref LPTIM_IT.
*               @arg LPTIM_IT_REPOK: repetition counter register update OK Interrupt source.
*               @arg LPTIM_IT_REPUE: Update event interrupt source.
*               @arg LPTIM_IT_DOWN: Counter direction change up to down Interrupt source.
*               @arg LPTIM_IT_UP: Counter direction change down to up Interrupt source.
*               @arg LPTIM_IT_ARROK: Autoreload register update OK Interrupt source.
*               @arg LPTIM_IT_CMPOK: Compare register update OK Interrupt source.
*               @arg LPTIM_IT_EXTTRIG: External trigger edge event Interrupt source.
*               @arg LPTIM_IT_ARRM: Autoreload match Interrupt source.
*               @arg LPTIM_IT_CMPM: Compare match Interrupt source.
*@param : NewState: new state of the interrupts sources.
*                   This parameter can be: ENABLE or DISABLE.
*@return: None
******************************************************************************/
void LPTIM_ITConfig(LPTIM_TypeDef* LPTIMx, uint32_t IT, FunctionalState NewState)
{
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));
    assert_param(IS_LPTIM_ALL_IT(IT));
    
    if (NewState != DISABLE)
        LPTIMx->IER |= IT;
    else
        LPTIMx->IER &= ~IT;
}

/******************************************************************************
*@brief : Checks whether the specified LPTIM flag is set or not.
*@param : LPTIMx: where x can be 1.
*@param : Flag: specifies the flag to check.
*               This parameter can be one of @ref LPTIM_Flags.
*                 @arg LPTIM_FLAG_REPOK: repetition counter register update OK Flag.
*                 @arg LPTIM_FLAG_REPUE: Update event Flag.
*                 @arg LPTIM_FLAG_DOWN: Counter direction change up to down Flag.
*                 @arg LPTIM_FLAG_UP: Counter direction change down to up Flag.
*                 @arg LPTIM_FLAG_ARROK: Autoreload register update OK Flag.
*                 @arg LPTIM_FLAG_CMPOK: Compare register update OK Flag.
*                 @arg LPTIM_FLAG_EXTTRIG: External trigger edge event Flag.
*                 @arg LPTIM_FLAG_ARRM: Autoreload match Flag.
*                 @arg LPTIM_FLAG_CMPM: Compare match Flag.
*@return: None
******************************************************************************/
FlagStatus LPTIM_GetFlagStatus(LPTIM_TypeDef* LPTIMx, uint32_t Flag)
{
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));
    assert_param(IS_LPTIM_SINGLE_FLAG(Flag));
    
    if (LPTIMx->ISR & Flag)
        return (SET);
    else
        return (RESET);
}

/******************************************************************************
*@brief : Clears the LPTIMx's pending flag.
*@param : LPTIMx: where x can be 1.
*@param : Flag: specifies the pending bit to clear.
*               This parameter can be combination of @ref LPTIM_Flags.
*                 @arg LPTIM_FLAG_REPOK: repetition counter register update OK Flag.
*                 @arg LPTIM_FLAG_REPUE: Update event Flag.
*                 @arg LPTIM_FLAG_DOWN: Counter direction change up to down Flag.
*                 @arg LPTIM_FLAG_UP: Counter direction change down to up Flag.
*                 @arg LPTIM_FLAG_ARROK: Autoreload register update OK Flag.
*                 @arg LPTIM_FLAG_CMPOK: Compare register update OK Flag.
*                 @arg LPTIM_FLAG_EXTTRIG: External trigger edge event Flag.
*                 @arg LPTIM_FLAG_ARRM: Autoreload match Flag.
*                 @arg LPTIM_FLAG_CMPM: Compare match Flag.
*@return: None
******************************************************************************/
void LPTIM_ClearFlag(LPTIM_TypeDef* LPTIMx, uint32_t Flag)
{
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));
    assert_param(IS_LPTIM_ALL_FLAG(Flag));
    
    LPTIMx->ICR = Flag;
}

/******************************************************************************
*@brief : Check whether the specified LPTIM interrupt has occurred or not.
*@param : LPTIMx: where x can be 1.
*@param : IT: specifies the LPTIM interrupt source to check.
*             This parameter can be one of @ref LPTIM_IT.
*               @arg LPTIM_IT_REPOK: repetition counter register update OK Interrupt source.
*               @arg LPTIM_IT_REPUE: Update event interrupt source.
*               @arg LPTIM_IT_DOWN: Counter direction change up to down Interrupt source.
*               @arg LPTIM_IT_UP: Counter direction change down to up Interrupt source.
*               @arg LPTIM_IT_ARROK: Autoreload register update OK Interrupt source.
*               @arg LPTIM_IT_CMPOK: Compare register update OK Interrupt source.
*               @arg LPTIM_IT_EXTTRIG: External trigger edge event Interrupt source.
*               @arg LPTIM_IT_ARRM: Autoreload match Interrupt source.
*               @arg LPTIM_IT_CMPM: Compare match Interrupt source.
*@return: The new state of LPTIM_IT (SET or RESET).
******************************************************************************/
FlagStatus LPTIM_GetITStatus(LPTIM_TypeDef* LPTIMx, uint32_t IT)
{
    /* Check the parameters */
    assert_param(IS_LPTIM_ALL_PERIPH(LPTIMx));
    assert_param(IS_LPTIM_SINGLE_IT(IT));
    
    if ((LPTIMx->IER & IT) && (LPTIMx->ISR & IT))
        return (SET);
    else
        return (RESET);
}



