
/******************************************************************************
*@file  : spl_wdt.c
*@brief : WDT SPL module driver.
******************************************************************************/

#include "acm32g103_spl_conf.h" 


/******************************************************************************
*@brief : Deinitializes the WDG peripheral registers to their default reset values.
*@param : None.
*@return: None
******************************************************************************/
void WDT_DeInit(void)
{
    __RCC_WDT_CLK_ENABLE();
    __RCC_WDT_CLK_DISABLE();
}

/******************************************************************************
*@brief : Enables or disables the specified WDT.
*@param : NewState: specifies the new state of the WDT.
*                   This parameter can be: ENABLE or DISABLE.
*@return: None
******************************************************************************/
void WDT_Cmd(FunctionalState NewState)
{
    /* Check the parameters */
    assert_param(IS_WDT_FUNCTIONAL_STATE(NewState));
    
    if (NewState != DISABLE)
        WDT->CTRL |= WDT_CTRL_EN;
    else
        WDT->CTRL &= ~WDT_CTRL_EN;
}

/******************************************************************************
*@brief : Set the WDT prescaler.
*@param : Prescaler: specifies the WDG Prescaler value.
*                    This parameter can be one of @ref WDT_Clock_Prescaler.
*                      @arg WDT_CLOCK_PRESCALER_1: WDG prescaler set to 1.
*                      @arg WDT_CLOCK_PRESCALER_2: WDG prescaler set to 2.
*                      @arg WDT_CLOCK_PRESCALER_4: WDG prescaler set to 4.
*                      @arg WDT_CLOCK_PRESCALER_8: WDG prescaler set to 8.
*                      @arg WDT_CLOCK_PRESCALER_16: WDG prescaler set to 16.
*                      @arg WDT_CLOCK_PRESCALER_32: WDG prescaler set to 32.
*                      @arg WDT_CLOCK_PRESCALER_64: WDG prescaler set to 64.
*                      @arg WDT_CLOCK_PRESCALER_128: WDG prescaler set to 128.
*@return: None
******************************************************************************/
void WDT_SetPrescaler(uint32_t Prescaler)
{
    /* Check the parameters */
    assert_param(IS_WDT_PRESCALER(Prescaler));
    
    WDT->CTRL = (WDT->CTRL & ~WDT_CTRL_DIVISOR) | Prescaler;
}

/******************************************************************************
*@brief : Set the WDT mode.
*@param : Prescaler: specifies the WDG mode.
*                    This parameter can be one of @ref WDT_Mode.
*                      @arg WDT_MODE_RESET: reset mode.
*                      @arg WDT_MODE_INTERRUPT: interrupt mode.
*@return: None
******************************************************************************/
void WDT_SetMode(uint32_t Mode)
{
    /* Check the parameters */
    assert_param(IS_WDT_MODE(Mode));
    
    WDT->CTRL = (WDT->CTRL & ~WDT_CTRL_MODE) | Mode;
}

/******************************************************************************
*@brief : Enables or disables the interrupt.
*@param : NewState: specifies the new state of the interrupt.
*                   This parameter can be: ENABLE or DISABLE.
*@return: None
******************************************************************************/
void WDT_ITCmd(FunctionalState NewState)
{
    /* Check the parameters */
    assert_param(IS_WDT_FUNCTIONAL_STATE(NewState));
    
    if (NewState != DISABLE)
        WDT->CTRL |= WDT_CTRL_INTEN;
    else
        WDT->CTRL &= ~WDT_CTRL_INTEN;
}

/******************************************************************************
*@brief : Set the WDT Load value.
*@param : LoadValue: specifies the WDT Load value.
*                    This parameter must be a number between 0 and 0xFFFFFFFF.
*@return: None
******************************************************************************/
void WDT_SetLoadValue(uint32_t LoadValue)
{
    WDT->LOAD = LoadValue;
}

/******************************************************************************
*@brief : Get the WDT Counter value.
*@param : None.
*@return: the WDT Load value.
******************************************************************************/
uint32_t WDT_GetCounterValue(void)
{
    return (WDT->COUNT);
}


/******************************************************************************
*@brief : Set the Interrupt clear time limit.
*@param : ITClrTime: specifies the Interrupt clear time limit.
*                    This parameter must be a number between 0 and 0xFFFF.
*@return: None.
******************************************************************************/
void WDT_SetITClrTime(uint32_t ITClrTime)
{
    /* Check the parameters */
    assert_param(IS_WDT_INTCLRTIME(ITClrTime));
    
    WDT->INTCLRTIME = (WDT->INTCLRTIME & ~WDT_INTCLRTIME_INTCLRT) | ITClrTime;
}

/******************************************************************************
*@brief : Feed dog.
*@param : None.
*@return: None.
******************************************************************************/
void WDT_FeedDog(void)
{
    WDT->FEED = WDT_CMD_FEED_WATCHDOG;
}

/******************************************************************************
*@brief : Checks whether the specified WDT flag is set or not.
*@param : Flag: specifies the flag to check.
*               This parameter can be a value of @ref WDT_Flag.
*                 @arg WDT_FLAG_RIS: raw interrupt status.
*@return: The new state of IWDT_Flag (SET or RESET).
******************************************************************************/
FlagStatus WDT_GetFlagStatus(uint32_t Flag)
{
    /* Check the parameters */
    assert_param(IS_WDT_FLAGS(Flag));

    if (WDT->RIS & Flag)
        return (SET);
    else
        return (RESET);
}





