/*
  ******************************************************************************
  * @file    app.c
  * @version V1.0.0
  * @date    2021
  * @brief   DMA demo source code.
  ******************************************************************************
*/
#include "app.h"

#define BUFFER_LENGTH    (256)

uint8_t TxBuffer[BUFFER_LENGTH];
uint8_t RxBuffer[BUFFER_LENGTH];

SPI_Handler_t hspi;

void printf_buff_byte(uint8_t* buff, uint32_t length)
{
	uint32_t i;

	for(i = 0; i < length; i++)
	{
		printf("%.2x ",buff[i]);	
	}
	printf("\n");
}

void SPI1_GPIO_Init(void)
{
    GPIO_InitTypeDef gpio;
    __RCC_GPIOA_CLK_ENABLE();
    
    /*  PA4    CS        AF0 
        PA5    SCK       AF0       
        PA12   MOSI      AF0
        PA11   MISO      AF0
        PA7    IO2(WP)   AF3
        PA6    IO3(HOLD) AF3
    */
    
    gpio.Pin = GPIO_PIN_4 | GPIO_PIN_5;
    gpio.Mode = GPIO_MODE_AF_PP;
    gpio.Pull = GPIO_PULLUP;
    gpio.Drive = GPIO_DRIVE_LEVEL3;
    gpio.Alternate = GPIO_FUNCTION_0;
    GPIO_Init(GPIOA, &gpio);
    
    gpio.Pin = GPIO_PIN_11 | GPIO_PIN_12 ;
    gpio.Mode = GPIO_MODE_AF_PP;
    gpio.Pull = GPIO_PULLUP;
    gpio.Drive = GPIO_DRIVE_LEVEL3;
    gpio.Alternate = GPIO_FUNCTION_0;
    GPIO_Init(GPIOA, &gpio);
    
    printfS("SPI1 PINs selected:\r\n");
    printfS("SPI1 CS  :PA4\r\n");
    printfS("SPI1 SCK :PA5\r\n");
    printfS("SPI1 MOSI:PA12\r\n");
    printfS("SPI1 MISO:PA11\r\n");
    
    gpio.Pin = GPIO_PIN_6 | GPIO_PIN_7;
    gpio.Mode = GPIO_MODE_AF_PP;
    gpio.Pull = GPIO_PULLUP;
    gpio.Drive = GPIO_DRIVE_LEVEL3;
    gpio.Alternate = GPIO_FUNCTION_3;
    
    GPIO_Init(GPIOC, &gpio);
    printfS("SPI1 WP  :PA7\r\n");
    printfS("SPI1 HOLD:PA6\r\n");    
}


void SPI1_Init(void)
{
    SPI_InitTypeDef SPI_InitStruct;
    
    __RCC_SPI1_CLK_ENABLE();
    
    SPI_InitStruct.SPI_Mode = SPI_MODE_SLAVE;
                  
    SPI_InitStruct.SPI_Work_Mode = SPI_WORK_MODE_0;
                  
    SPI_InitStruct.X_Mode = SPI_1X_MODE;      
                  
    SPI_InitStruct.First_Bit = SPI_FIRSTBIT_MSB;
                  
    SPI_InitStruct.Slave_SWCS = SPI_SLAVE_SWCS_DISABLE;
    
    SPI_InitStruct.Slave_CS_Fliter = SPI_SLAVE_CS_FILTER_DISABLE;
    
    SPI_InitStruct.Slave_SCLK_Fliter = SPI_SLAVE_SCLK_FILTER_DISABLE;
   
    SPI_Init(SPI1, &SPI_InitStruct);
    
    NVIC_ClearPendingIRQ(SPI1_IRQn);
    
    NVIC_EnableIRQ(SPI1_IRQn);
}
/************************************************************************
 * function   : SPI_Slave_Comm_Test
 * Description: SPI Slave Communicate Test.
 ************************************************************************/ 
void SPI_Slave_Comm_Test(void)
{
    uint32_t i;
    uint32_t rx_cnt = 0;
    uint32_t tx_cnt = 0;
    
    SPI1_GPIO_Init();
    
    SPI1_Init();
    
    printfS("SPI Slave Comm Polling Test\r\n");
    
    while (1) 
    {                           
        rx_cnt = SPI_Slave_Receive(SPI1, RxBuffer, sizeof(RxBuffer), 10000);

        if(rx_cnt)
        {
            tx_cnt = SPI_Slave_Transmit(SPI1, RxBuffer, rx_cnt, 0);
        
            printf("Rx_Count = %d. Tx_Count = %d.\r\n", rx_cnt, tx_cnt);
            for (i = 0; i < BUFFER_LENGTH; i++)
            {
                RxBuffer[i] = 0;
            }
        }
    }
}

/************************************************************************
 * function   : SPI_Slave_Comm_IT
 * Description: SPI Slave Communicate Interrupt Test.
 ************************************************************************/ 
void SPI_Slave_Comm_IT(void)
{
    uint32_t i;

    SPI1_GPIO_Init();
    
    SPI1_Init();

    printfS("SPI Slave Comm IT Test\r\n");

    while (1) 
    {
        SPI_Slave_Receive_IT(SPI1, &hspi, RxBuffer, sizeof(RxBuffer));
        while(hspi.RxState != SPI_RX_STATE_IDLE);

        SPI_Slave_Transmit_IT(SPI1, &hspi, RxBuffer, hspi.Rx_Count);
        while(hspi.TxState != SPI_TX_STATE_IDLE);
        
        printf("Rx_Count = %d. Tx_Count = %d.\r\n", hspi.Rx_Count, hspi.Tx_Count);
        for (i = 0; i < BUFFER_LENGTH; i++)
        {
            RxBuffer[i] = 0;
        }
    }
}

/************************************************************************
 * function   : SPI_Slave_DMA_Test
 * Description: SPI Slave DMA Test.
 ************************************************************************/ 
bool flag_dma_tx;
bool flag_dma_rx;

void SPI_Slave_DMA_Test(void)
{
    uint32_t i;
    
    DMA_InitTypeDef InitData;
    
    uint32_t DataLength = 50;
    
    SPI1_GPIO_Init();
    
    SPI1_Init();
    
    SPI_Slave_SWCS_Config(SPI1, SPI_Slave_SWCS_Set);
    SPI_Slave_SWCS_Config(SPI1, SPI_Slave_SWCS_Reset);
    
    memset((void*)&InitData, 0, sizeof(InitData));
    
    __RCC_DMA1_CLK_ENABLE();
    
    InitData.Mode        = DMA_MODE_NORMAL;
    InitData.DataFlow    = DMA_DATAFLOW_M2P;
    InitData.ReqID       = DMA1_REQ1_SPI1_SEND;
    InitData.SrcAddr     = 0;
    InitData.DestAddr    = 0;
    InitData.Size        = 0;
    InitData.RawInt      = DMA_RAWINT_ENABLE;
    InitData.SrcWidth    = DMA_SRCWIDTH_BYTE;
    InitData.DestWidth   = DMA_DESTWIDTH_BYTE;
    InitData.SrcInc      = DMA_SRCINC_ENABLE;
    InitData.DestInc     = DMA_DESTINC_DISABLE;
    InitData.SrcBurst    = DMA_SRCBURST_1;
    InitData.DestBurst   = DMA_DESTBURST_1;
       
    DMA_Init(DMA1_Channel0, &InitData);
    
    memset((void*)&InitData, 0, sizeof(InitData));
    
    InitData.Mode        = DMA_MODE_NORMAL;
    InitData.DataFlow    = DMA_DATAFLOW_P2M;
    InitData.ReqID       = DMA1_REQ2_SPI1_RECV;
    InitData.SrcAddr     = 0;
    InitData.DestAddr    = 0;
    InitData.Size        = 0;
    InitData.RawInt      = DMA_RAWINT_ENABLE;
    InitData.SrcWidth    = DMA_SRCWIDTH_BYTE;
    InitData.DestWidth   = DMA_DESTWIDTH_BYTE;
    InitData.SrcInc      = DMA_SRCINC_DISABLE;
    InitData.DestInc     = DMA_DESTINC_ENABLE;
    InitData.SrcBurst    = DMA_SRCBURST_1;
    InitData.DestBurst   = DMA_DESTBURST_1;
       
    DMA_Init(DMA1_Channel1, &InitData);

    SPI_DMA_RXFIFO_Level_Set(SPI1, SPI_RX_CTL_DMA_LEVEL_0);
    SPI_DMA_TXFIFO_Level_Set(SPI1, SPI_TX_CTL_DMA_LEVEL_0);
    DMA_ITConfig(DMA1_Channel1, DMA_CHCONFIG_ITC | DMA_CHCONFIG_IE , ENABLE);
    DMA_ITConfig(DMA1_Channel0, DMA_CHCONFIG_ITC | DMA_CHCONFIG_IE , ENABLE);

    NVIC_ClearPendingIRQ(DMA1_IRQn);
    NVIC_SetPriority(DMA1_IRQn, 5);
    NVIC_EnableIRQ(DMA1_IRQn); 
    
    printfS("SPI Slave Comm DMA Test\r\n");
    
    while (1) 
    {
        SPI_ClearFlag(SPI1, SPI_STATUS_BATCH_DONE);

        SPI_ITConfig(SPI1, SPI_IE_RX_BATCH_DONE_EN, ENABLE);

        SPI_WriteBatch(SPI1, DataLength);

        SPI_RxCmd(SPI1, ENABLE);
        
        SPI_ClearRxFifo(SPI1);

        SPI_DMACmd(SPI1, SPI_DMAReq_RX, ENABLE);    

        DMA_SrcAddrConfig(DMA1_Channel1, (uint32_t)&SPI1->DAT);

        DMA_DestAddrConfig(DMA1_Channel1, (uint32_t)RxBuffer);

        DMA_SetTransferSize(DMA1_Channel1, DataLength);

        DMA_Cmd(DMA1_Channel1, ENABLE);        
        
        while (!flag_dma_rx);
        flag_dma_rx = false;       
        
        SPI_ClearFlag(SPI1, SPI_STATUS_BATCH_DONE);

        SPI_ITConfig(SPI1, SPI_IE_TX_BATCH_DONE_EN, ENABLE);
        
        SPI_WriteBatch(SPI1, DataLength);

        SPI_ClearTxFifo(SPI1);

        SPI_TxCmd(SPI1, ENABLE);

        SPI_DMACmd(SPI1, SPI_DMAReq_TX, ENABLE);    
        
        DMA_SrcAddrConfig(DMA1_Channel0, (uint32_t)RxBuffer);
        
        DMA_DestAddrConfig(DMA1_Channel0, (uint32_t)&SPI1->DAT);
        
        DMA_SetTransferSize(DMA1_Channel0, DataLength);
            
        DMA_Cmd(DMA1_Channel0, ENABLE);
        
        
        while (!flag_dma_tx);
        flag_dma_tx = false;
        
        printf_buff_byte(RxBuffer, DataLength);
        
        for (i = 0; i < BUFFER_LENGTH;  i++)
        {
            RxBuffer[i] = 0;
        }
    }
}

/************************************************************************
 * function   : SPI_Slave_FullDuplex_Test
 * Description: SPI Slave Full Duplex Test. 
 ************************************************************************/ 
void SPI_Slave_FullDuplex_Test(void)
{
    uint32_t i;
    
    uint32_t COM_OK  = 0;
    uint32_t COM_Err = 0;
    uint32_t DataLength = 50;

    SPI1_GPIO_Init();
    
    SPI1_Init();
    
    printfS("SPI Slave Full Duplex Test is Ready!!! \r\n");
    
    while (1) 
    {
        for (i = 0; i < BUFFER_LENGTH; i++)
        {
            TxBuffer[i] = i+1;
            RxBuffer[i] = 0;
        }
                                
        SPI_Slave_TransmitReceive(SPI1, TxBuffer, RxBuffer, DataLength, 0);
        
        for (i = 0; i < DataLength; i++)
        {
            if (TxBuffer[i] != (RxBuffer[i]+1)) 
            {
                printfS("There is one mistake : TxBuffer[%d]: 0x%02x != RxBuffer[%d]: 0x%02x \r\n", i,TxBuffer[i], i, RxBuffer[i]);
                
                COM_Err++;
            }
            else 
            {
                COM_OK++;
            }
        }
        
        printfS("SPI Slave Full Duplex Test OK count %d times \r\n",  COM_OK);
        printfS("SPI Slave Full Duplex Test Err count %d times \r\n", COM_Err);
        COM_Err = 0;
        COM_OK = 0;
    }
}

/************************************************************************
 * function   : SPI_Slave_Test
 * Description: SPI Slave Test.
 ************************************************************************/ 
void SPI_Slave_Test(enum_TEST_MODE_t fe_Mode)
{
    printfS("---------- SPI Slave Test ----------\r\n");
        
    switch (fe_Mode)
    {
        case TEST_SLAVE_COMM: 
            SPI_Slave_Comm_Test();
            break; 
        
        case TEST_SLAVE_IT: 
            SPI_Slave_Comm_IT();
            break; 
        
        case TEST_SLAVE_DMA: 
            SPI_Slave_DMA_Test();
            break;
       
        case TEST_SLAVE_FULL_DUPLEX: 
            SPI_Slave_FullDuplex_Test();
            break;
       
        default: break;     
    }
}
